! ******************************************************************** !
! ATCHEM2 -- MODULE customRateFunctions
!
! This module contains user-defined functions that can be referenced
! in the mechanism file
! ******************************************************************** !
module custom_functions_mod
  implicit none

contains

  pure function VOCuptk1stOrd( temp, m, mass, rp, asa, gamma ) result ( k )
    use types_mod
    !
    ! Computes the rxn rate [1/s]for 1st-order uptake of dimer species.
    ! Note that asa and rp should be in m2/m3 and m, respectively. They are converted to cm units within the function.
    ! mass should be in units of g mol-1
    !
    real(kind=DP), intent(in) :: temp, m, mass, rp, asa, gamma 
    real(kind=DP) :: srMw, srTemp, convRp, convASA, dfkg
    real(kind=DP) :: k              

    srMw = mass**0.5_DP
    srTemp = temp**0.5_DP

    convRp = rp * 100.0_DP ! m to cm
    convASA = asa * 1E-2   ! m2/m3 to cm2/cm3

    ! DFKG = Gas phase diffusion coeff [cm2/s] (order of 0.1)
    dfkg = ( 9.45E+17_dp / m ) * srTemp *                              &
           ( 3.472E-2_dp + 1.0_dp / ( srMw * srMw ) )**0.5_DP
    !
    ! Compute ArsL1k according to the formula listed above
    k = convASA / ( (convRp / dfkg) + 2.749064E-4_dp * srMw / (gamma * srTemp) )
    
    return
  end function VOCuptk1stOrd


  ! -----------------------------------------------------------------
  ! The condensation half of gas-particle partitioning, equation taken
  ! from D'Ambro et al. 2017.
  pure function condenseRxn( temp, mass, rp, dg, alpha, asa) result ( condrate )
    use types_mod

    real(kind=DP), intent(in) :: temp, mass, rp, dg, alpha, asa
    real(kind=DP) :: condrate
    real(kind=DP) :: convmass, molspd, kmt

    convmass = mass / 1000.0_DP
    molspd = (( 3.0_DP * 8.314_DP * temp ) / convmass)**(0.5_DP)

    kmt = ((rp/dg)+(4.0_DP/(alpha*molspd)))**(-1.0_DP)
    
    condrate = kmt * asa
    return
  end function condenseRxn

  ! -----------------------------------------------------------------
  ! The evaporation half of gas-particle partitioning, equation taken
  ! from D'Ambro et al. 2017.
  pure function evapRxn( temp, mass, rp, dg, alpha, asa, vappress, coa) result ( evaprate )
    use types_mod

    real(kind=DP), intent(in) :: temp, mass, rp, dg, alpha, asa, vappress, coa
    real(kind=DP) :: evaprate
    real(kind=DP) :: convmass, molspd, kmt, cstar

    convmass = mass / 1000.0_DP
    molspd = (( 3.0_DP * 8.314_DP * temp ) / convmass)**(0.5_DP)

    kmt = ((rp/dg)+(4.0_DP/(alpha*molspd)))**(-1.0_DP)
    
    ! Convert the vapour pressure to a saturation concentration (in ug m-3)
    cstar = (vappress/(8.314_DP * temp)) * mass * 1.0E+06_DP

    evaprate = kmt * asa * (cstar/coa)
    return
  end function evapRxn

  pure function ARRPLUS_abde( a0, b0, d0, e0, TEMP ) result( k )
    use types_mod
    ! Modified Arrhenius law, skipping computation of ( T/300 )**c0,
    ! which evaluates to 1 when c0=0.  This avoids excess CPU cycles.
    ! (bmy, 12/18/20)
    !
    ! Used to compute the rate for these reactions:
    !    IHOO1 + HO2 = 0.063MVK + 0.063OH + 0.063HO2 + 0.063CH2O + 0.937RIPA
    !    IHOO1 + HO2 = RIPC
    !    IHOO4 + HO2 = 0.063MACR + 0.063OH + 0.063HO2 + 0.063CH2O + 0.937RIPB
    !    IHOO4 + HO2 = RIPD
    !    IHOO1       = CH2O + OH + MVK
    !    IHOO4       = MACR + OH + CH2O
    !
    real(kind=DP), intent(in) :: a0, b0, d0, e0, TEMP
    real(kind=DP) :: k
    !
    k = a0 * ( d0 + ( TEMP * e0 ) ) * EXP( -b0 / TEMP )
    k = MAX( k, 0.0_dp )
  end function ARRPLUS_abde
  
  pure function ARRPLUS_ade( a0, d0, e0, TEMP ) result( k )
    use types_mod
    ! Modified Arrhenius law, skipping computation of EXP( -b0/T )
    ! and ( 300/T )**c0 terms, which evaluate to 1 when b0 = c0 = 0.
    ! This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    ! Used to compute the rate for these reactions:
    !    IHOO1 + IHOO1 = 2MVK  + 2HO2 + 2CH2O
    !    IHOO4 + IHOO4 = 2MACR + 2HO2 + 2CH2O
    !    IHOO1 + IHOO4 = MACR + MVK + 2HO2 + 2CH2O
    !    IHOO1 + IHOO1 = HO2 + HC5A + CO + OH +  MVKHP
    !    IHOO4 + IHOO4 = HO2 + HC5A + CO + OH +  MCRHP
    !    IHOO1 + IHOO4 = HO2 + HC5A + CO + OH +  0.5MVKHP + 0.5MCRHP
    !    IHOO1 + MO2   = MVK + 2HO2 + 2CH2O :
    !    IHOO1 + MO2   = CH2O + 0.5HC5A + 1.5HO2 + 0.5MVKHP + 0.5CO + 0.5OH
    !    IHOO4 + MO2   = MACR + 2HO2 + 2CH2O
    !    IHOO4 + MO2   = CH2O + 0.5HC5A + 1.5HO2 +  0.5MCRHP + 0.5CO + 0.5OH
    !
    real(kind=DP), intent(in) :: a0, d0, e0, TEMP
    real(kind=DP) :: k
    !
    k = a0 * ( d0 + ( TEMP * e0 ) )
    k = MAX( k, 0.0_dp )
  end function ARRPLUS_ade

  pure function GCARR_ab( a0, b0, TEMP ) result( k )
    use types_mod
    ! Arrhenius function, skipping computation of EXP( c0/T ),
    ! which evaluates to 1 when c0=0.  This avoids excess CPU
    ! cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, b0, TEMP
    real(kind=DP) :: k, K300_OVER_TEMP
    !
    K300_OVER_TEMP = 300.0_DP / TEMP
    k = a0 * K300_OVER_TEMP**b0
  end function GCARR_ab
  
  pure function GCARR_abc( a0, b0, c0, TEMP ) result( k )
    use types_mod
    ! Arrhenius function, using all 3 terms.
    ! Use this when a0, b0, c0 are all nonzero.
    !
    real(kind=DP), intent(in) :: a0, b0, c0, TEMP
    real(kind=DP) :: k, K300_OVER_TEMP
    !
    K300_OVER_TEMP = 300.0_DP / TEMP
    k = a0 * EXP( c0 / TEMP ) * K300_OVER_TEMP**b0
  end function GCARR_abc
  
  pure function GCARR_ac( a0, c0, TEMP ) result( k )
    use types_mod
    ! Arrhenius function, skipping computation of ( 300/T )**b0,
    ! which evaluates to 1 when b0=0.  This avoids excess CPU
    ! cycles (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, TEMP
    real(kind=DP) :: k
    !
    k = a0 * EXP( c0 / TEMP )
  end function GCARR_ac
  
  pure function GCJPLEQ_acabab( a0, c0, a1, b1, a2, b2, fv, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Calculates the equilibrium constant
    ! Find the backwards reaction by K=kforward/kbackwards
    ! Calculates the rate constant of the forward reaction
    !
    ! Used to compute the rate for these reactions:
    !    PPN        = RCO3 + NO2
    !    PAN        = MCO3 + NO2
    !    ClOO  {+M} = Cl   + O2 {+M}
    !    Cl2O2 {+M} = 2ClO      {+M}
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0
    !    EXP(c1/T)
    !    EXP(c2/T)
    ! because b0 = c1 = c2 = 0.  Therefore we can skip computing these terms.
    ! Also, fct1 = fct2 = 0, so we will skip those terms as well.  This is
    ! more computationally efficient. (bmy, 1/25/20)
    !
    real(kind=DP), intent(in) :: a0, c0, a1, b1, a2, b2, fv, TEMP, NUMDEN
    real(kind=DP) :: k0, k1, k
    !
    k0 = a0 * EXP( c0 / TEMP )               ! backwards rxn rate
    k1 = GCJPLPR_abab( a1, b1, a2, b2, fv, TEMP, NUMDEN )  ! forwards rxn rate
    k  = k1 / k0
  end function GCJPLEQ_acabab
  
  pure function GCJPLPR_aa( a1, a2, fv, NUMDEN ) result( k )
    use types_mod
    ! Third body effect for pressure dependence of rate coefficients.
    ! a1 is Arrhenius parameters for the lower-limit rate.
    ! a2 is Arrhenius parameters for the upper-limit rate.
    ! fv is the falloff curve paramter, (see ATKINSON ET. AL (1992)
    ! J. Phys. Chem. Ref. Data 21, P. 1145). Usually fv = 0.6.
    !
    ! Used to compute the rate for this reaction:
    !    Cl + PRPE {+M} = HCl + PO2 {+M}
    !
    ! For this reactions, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b1 * EXP(c1/T)
    !    (300/T)**b2 * EXP(c2/T)
    ! because b1 = b2 = c1 = c2 = 0.  Therefore we can skip computing
    ! these terms.  Also, fct1 = fct2 = 0, so we will skip computing
    ! these terms as well.  This is more computationally efficient.
    ! (bmy, 1/25/20)
    !
    real(kind=DP), intent(in) :: a1,   a2,    fv, NUMDEN
    real(kind=DP) :: rlow, xyrat, blog, fexp, k
    !
    rlow  = a1 * NUMDEN
    xyrat = rlow / a2         ! rhigh = a2
    blog  = LOG10( xyrat )
    fexp  = 1.0_dp / ( 1.0_dp + ( blog * blog ) )
    k     = rlow * ( fv**fexp ) / ( 1.0_dp + xyrat )
  end function GCJPLPR_aa
  
  pure function GCJPLPR_aba( a1, b1, a2, fv, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Third body effect for pressure dependence of rate coefficients.
    ! a1, b1 are the Arrhenius parameters for the lower-limit rate.
    ! a2     is  the Arrhenius parameters for the upper-limit rate.
    ! fv     is the falloff curve paramter, (see ATKINSON ET. AL (1992)
    !        J. Phys. Chem. Ref. Data 21, P. 1145). Usually fv = 0.6.
    !
    ! Used to compute the rate for these reactions:
    !    OH  + OH  {+M} = H2O2
    !    NO2 + OH  {+M} = HNO3       {+M}
    !    Cl  + O2  {+M} = ClOO       {+M}
    !    SO2 + OH  {+M} = SO4  + HO2
    !    Br  + NO2 {+M} = BrNO2      {+M}
    !    NO  + O   {+M} = NO2        {+M}
    !    I   + NO2 {+M} = IONO       {+M}
    !    I   + NO  {+M} = INO        {+M}
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    EXP(c1/T)
    !    (300/T)**b2 * EXP(c2/T)
    ! because b2 = c1 = c2 = 0.  Therefore we can skip computing these
    ! terms.  Also, fct1 = fct2 = 0, so we will skip computing these
    ! terms as well.  This is more computationally efficient.
    ! (bmy, 1/25/20)
    !
    real(kind=DP), intent(in) :: a1,   b1,    a2,   fv, TEMP, NUMDEN
    real(kind=DP) :: rlow, xyrat, blog, fexp, k, K300_OVER_TEMP
    !
    K300_OVER_TEMP = 300.0_DP / TEMP
    rlow  = a1 * ( K300_OVER_TEMP**b1 ) * NUMDEN
    xyrat = rlow / a2                                  ! rhigh = a2
    blog  = LOG10( xyrat )
    fexp  = 1.0_dp / ( 1.0_dp + ( blog * blog ) )
    k     = rlow * ( fv**fexp ) / ( 1.0_dp + xyrat )
  end function GCJPLPR_aba

  pure function GCJPLPR_abab( a1, b1, a2, b2, fv, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Third body effect for pressure dependence of rate coefficients.
    ! a1, b1 are the Arrhenius parameters for the lower-limit rate.
    ! a2, b2 are the Arrhenius parameters for the upper-limit rate.
    ! fv     is the falloff curve paramter, (see ATKINSON ET. AL (1992)
    !        J. Phys. Chem. Ref. Data 21, P. 1145). Usually fv = 0.6.
    !
    ! Used to compute the rate for these reactions:
    !    NO   + OH  {+M} = HNO2  {+M}
    !    HO2  + NO2 {+M} = HNO4
    !    NO2  + NO3 {+M} = N2O5
    !    ClO  + NO2 {+M} = ClNO3 {+M}
    !    MCO3 + NO2 {+M} = PAN
    !    RCO3 + NO2 {+M} = PPN
    !    PRPE + OH  {+M} = PO2
    !    MO2  + NO2 {+M} = MPN   {+M}
    !    BrO  + NO2 {+M} = BrNO3 {+M}
    !    NO2  + O   {+M} = NO3   {+M}
    !    H    + O2  {+M} = HO2   {+M}
    !    IO   + NO2 {+M} = IONO2 {+M}
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    EXP(c1/T)
    !    EXP(c2/T)
    ! because c1 = c2 = 0.  Therefore we can skip computing these
    ! terms.  Also, fct1 = fct2 = 0, so we will skip computing these
    ! terms as well.  This is more computationally efficient.
    ! (bmy, 1/25/20)
    !
    real(kind=DP), intent(in) :: a1,   b1,    a2,    b2,   fv, TEMP, NUMDEN
    real(kind=DP) :: rlow, rhigh, xyrat, blog, fexp, k, K300_OVER_TEMP
    !
    K300_OVER_TEMP = 300.0_DP / TEMP
    rlow  = a1 * ( K300_OVER_TEMP**b1 ) * NUMDEN
    rhigh = a2 * ( K300_OVER_TEMP**b2 )
    xyrat = rlow / rhigh
    blog  = LOG10( xyrat )
    fexp  = 1.0_dp / ( 1.0_dp + ( blog * blog ) )
    k     = rlow * ( fv**fexp ) / ( 1.0_dp + xyrat )
  end function GCJPLPR_abab
  
  pure function GCJPLPR_abcabc( a1, b1, c1, a2, b2, c2, fv, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Third body effect for pressure dependence of rate coefficients.
    ! a1, b1, c1 are the Arrhenius parameters for the lower-limit rate.
    ! a2, b2, c2 are the Arrhenius parameters for the upper-limit rate.
    ! fv         is the falloff curve paramter, (see ATKINSON ET. AL (1992)
    !           J. Phys. Chem. Ref. Data 21, P. 1145). Usually fv = 0.6.
    !
    ! Used to compute the rate for these reactions:
    !    HNO4 {+M} = HO2 + NO2
    !    N2O5 {+M} = NO2 + NO3
    !    MPN  {+M} = MO2 + NO2
    !
    real(kind=DP), intent(in) :: a1,   b1,    c1,    a2,   b2,   c2,  fv, TEMP, NUMDEN
    real(kind=DP) :: rlow, rhigh, xyrat, blog, fexp, k, K300_OVER_TEMP
    !
    K300_OVER_TEMP = 300.0_DP / TEMP
    rlow  = a1 * ( K300_OVER_TEMP**b1 ) * EXP( c1 / TEMP ) * NUMDEN
    rhigh = a2 * ( K300_OVER_TEMP**b2 ) * EXP( c2 / TEMP )
    xyrat = rlow / rhigh
    blog  = LOG10( xyrat )
    fexp  = 1.0_dp / ( 1.0_dp + ( blog * blog ) )
    k     = rlow * ( fv**fexp ) / ( 1.0_dp + xyrat )
  end function GCJPLPR_abcabc
  
  pure function GC_ALK( a0, b0, c0, n, x0, y0, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Used to compute the rate for these reactions:
    !   IHOO1    + NO =      NO2 + ...
    !   IHOO4    + NO =      NO2 + ...
    !   IHP001   + NO =      NO2 + ...
    !   IHP002   + NO =      NO2 + ...
    !   IHP003   + NO =      NO2 + ...
    !   IEPOXAOO + NO =      NO2 + ...
    !   IEPOXBOO + NO =      NO2 + ...
    !   ICHOO    + NO =      NO2 + ...
    !   ISOPNOO1 + NO = 1.728NO2 + ...
    !   ISOPNOO2 + NO =      NO2 + ...
    !   IDHNDOO1 + NO =      NO2 + ...
    !   IDHNDOO2 + NO =      NO2 + ...
    !   IDHNBOO  + NO =      NO2 + ...
    !   IDHNDOO  + NO =      NO2 + ...
    !   INO2B    + NO = 2.000NO2 + ...
    !   INO2D    + NO =      NO2 + ...
    !   IHPNBOO  + NO = 1.065NO2 + ...
    !   IHPNDOO  + NO =      NO2 + ...
    !   MVKOHOO  + NO =      NO2 + ...
    !   MCROHOO  + NO =      NO2 + ...
    !
    real(kind=DP), intent(in) :: a0, b0, c0, n,  x0, y0, TEMP, NUMDEN
    real(kind=DP) :: k0, k1, k2, k3, k4, k
    !
    k0 = 2.0E-22_dp * EXP( n )
    k1 = 4.3E-1_dp * ( TEMP / 298.0_dp)**(-8)
    k0 = k0 * NUMDEN
    k1 = k0 / k1
    k2 = ( K0 / ( 1.0_dp +K1 ) )                                             &
       * 4.1E-1_dp**( 1.0_dp / ( 1.0_dp + ( LOG10( k1 ) )**2) )
    k3 = c0/ ( k2 + c0 )
    k4 = a0 * ( x0 - TEMP*y0 )
    k  = k4 * EXP( b0 / TEMP ) * k3
    k  = MAX( k, 0.0_dp )
  end function GC_ALK
  
  pure function GC_DMSOH_acac( a0, c0, a1, c1, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Reaction rate for:
    !    DMS + OH = 0.750SO2 + 0.250MSA + MO2
    !
    ! For this reaction, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0
    !    (300/T)**b1
    ! Because b0 = b1 = 0.  Therefore we can skip computing these
    ! terms.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, a1, c1, TEMP, NUMDEN
    real(kind=DP) :: k0, k1, k
    !
    k0 = a0 * EXP( c0 / TEMP )
    k1 = a1 * EXP( c1 / TEMP )
    k  = ( k0 * NUMDEN * 0.2095e0_dp ) / ( 1.0_dp + k1 * 0.2095e0_dp )
  end function GC_DMSOH_acac
  
  pure function GC_EPO_a( a1, e1, m1, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Used to compute the rate for these reactions:
    !    RIPA   + OH = 0.67IEPOXA   + 0.33IEPOXB   + OH + 0.005LVOC
    !    RIPB   + OH = 0.68IEPOXA   + 0.321IEPOB   + OH + 0.005LVOC
    !    IEPOXA + OH = 0.67IEPOXA00 + 0.33IEPOXB00
    !    IEPOXB + OH = 0.81IEPOXA00 + 0.19IEPOXB00
    !    IHN2   + OH = 0.67IEPOXA   + 0.33IEPOXB   + NO2
    !    IHN3   + OH = 0.67IEPOXA   + 0.33IEPOXB   + NO2
    !    IHN1   + OH = IEPOXD       + NO2
    !    IHN4   + OH = IEPOXD       + NO2
    !    INPB   + OH = OH           + ITHN
    !    INPD   + OH = OH           + ITHN
    !    INPD   + OH = NO2          + ICHE
    !    ICN    + OH = NO2          + ICHE
    !
    real(kind=DP), intent(in) :: a1, e1, m1, TEMP, NUMDEN
    real(kind=DP) :: k1, k
    !
    k1 = 1.0_dp / ( m1 * NUMDEN + 1.0_dp )
    k  = a1 * EXP( e1 / TEMP ) *  K1
  end function GC_EPO_a
  
  pure function GC_GLYCOH_A_a( a0, TEMP ) result( k )
    use types_mod
    ! Used to compute the rate for this reaction:
    !    GLYC + OH = 0.732CH2O + 0.361CO2  + 0.505CO    + 0.227OH
    !              + 0.773HO2  + 0.134GLYX + 0.134HCOOH
    ! which is the "A" branch of GLYC + OH.
    !
    ! For this reaction, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0 * EXP(c0/T)
    ! Because b0 = c0 = 0.  Therefore we can skip computing these
    ! terms.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, TEMP
    real(kind=DP) :: glyc_frac, k
    real(kind=DP), PARAMETER  :: exp_arg = -1.0_dp / 73.0_dp
    !
    glyc_frac = 1.0_dp - 11.0729_dp * EXP( exp_arg * TEMP )
    glyc_frac = MAX( glyc_frac, 0.0_dp )
    k         = a0 * glyc_frac
  end function GC_GLYCOH_A_a
  
  pure function GC_GLYCOH_B_a( a0, TEMP ) result( k )
    use types_mod
    ! Used to compute the rate for this reaction:
    !    GLYC + OH = HCOOH + OH + CO
    ! which is the "B" branch of GLYC + OH.
    !
    ! For this reaction, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0 * EXP(c0/T)
    ! Because b0 = c0 = 0.  Therefore we can skip computing these
    ! terms.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, TEMP
    real(kind=DP) :: glyc_frac, k
    real(kind=DP), PARAMETER  :: exp_arg = -1.0_dp / 73.0_dp
    !
    glyc_frac = 1.0_dp - 11.0729_dp * EXP( exp_arg * TEMP )
    glyc_frac = MAX( glyc_frac, 0.0_dp )
    k         = a0 * ( 1.0_dp - glyc_frac )
  end function GC_GLYCOH_B_a
  
  pure function GC_GLYXNO3_ac( a0, c0, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Reaction rate for:
    !    GLYX + NO3 = HNO3 + HO2 + 2CO
    !    i.e. the HO2 + 2*CO branch
    !
    ! For this reaction, this Arrhenius term evaluates to 1:
    !    (300/T)**b0
    ! because b0 = 0.  Therefore we can skip computing this
    ! term.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, TEMP, NUMDEN
    real(kind=DP) :: O2, k
    !
    ! ---  K = K1*([O2]+3.5D18)/(2*[O2]+3.5D18)
    O2 = NUMDEN * 0.2095_dp
    k  = a0 * EXP( c0 / TEMP )
    k   = k * ( O2 + 3.5E+18_dp ) / ( 2.0_dp * O2 + 3.5E+18_dp )
  end function GC_GLYXNO3_ac
  
  pure function GC_HACOH_A_ac( a0, c0, TEMP ) result( k )
    use types_mod
    ! Used to compute the rate for this reaction:
    !    HAC + OH = MGLY + HO2
    ! which is the "A" branch of HAC + OH.
    !
    ! For this reaction, this Arrhenius law term evaluates to 1:
    !    (300/T)**b0
    ! because b0 = 0.  Therefore we can skip computing this
    ! term.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, TEMP
    real(kind=DP) :: k0, hac_frac, k
    real(kind=DP), PARAMETER  :: exp_arg = -1.0_dp / 60.0_dp
    !
    k0       = a0 * EXP( c0 / TEMP )
    hac_frac = 1.0_dp - 23.7_dp * EXP( exp_arg * TEMP )
    hac_frac = MAX( hac_frac, 0.0_dp )
    k        = k0 * hac_frac
  end function GC_HACOH_A_ac
  
  pure function GC_HACOH_B_ac( a0, c0, TEMP ) result( k )
    use types_mod
    ! Used to compute the rate for this reaction:
    !    HAC + OH = 0.5HCOOH + OH + 0.5ACTA + 0.5CO2 + 0.5CO + 0.5MO2
    ! which is the "B" branch of HAC + OH.
    !
    ! For this reaction, this Arrhenius law term evaluates to 1:
    !    (300/T)**b0}
    ! because b0 = 0.  Therefore we can skip computing this
    ! term.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, TEMP
    real(kind=DP) :: k0, hac_frac, k
    real(kind=DP), PARAMETER  :: exp_arg = -1.0_dp / 60.0_dp
    !
    k0       = a0 * EXP( c0 / TEMP )
    hac_frac = 1.0_dp - 23.7_dp * EXP( exp_arg * TEMP )
    hac_frac = MAX( hac_frac, 0.0_dp )
    k        = k0 * ( 1.0_dp - hac_frac )
  end function GC_HACOH_B_ac
  
  pure function GC_HO2HO2_acac( a0, c0, a1, c1, TEMP, NUMDEN, H2O ) result( k )
    use types_mod
    ! Used to compute the rate for these reactions:
    !    HO2 + HO2 = H2O2 + O2
    !
    ! For this reaction, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0
    !    (300/T)**b1
    ! because b0 = b1 = 0.  Therefore we can skip computing these
    ! terms.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, a1, c1, TEMP, NUMDEN, H2O
    real(kind=DP) :: k0, k1, k
    !
    k0 = a0 * EXP( c0 / TEMP )
    k1 = a1 * EXP( c1 / TEMP )
    k  = ( k0     + k1         * NUMDEN                           )          &
       * ( 1.0_dp + 1.4E-21_dp * H2O    * EXP( 2200.0_dp / TEMP ) )
  end function GC_HO2HO2_acac
  
  pure function GC_ISO1( a0, b0, c0, d0, e0, f0, g0, TEMP ) result( k )
    use types_mod
    ! Used to compute the rate for these reactions:
    !    ISOP + OH = LISOPOH + IHOO1
    !    ISOP + OH = LISOPOH + IHOO4
    !
    real(kind=DP), intent(in) :: a0, b0, c0, d0, e0, f0, g0, TEMP
    real(kind=DP) :: k0, k1, k2, k
    !
    k0 = d0 * EXP( e0 / TEMP ) * EXP( 1.0E8_dp / TEMP**3 )
    k1 = f0 * EXP( g0 / TEMP )
    k2 = c0 * k0 / ( k0 + k1 )
    k  = a0 * EXP( b0 / TEMP ) * ( 1.0_dp - k2 )
  end function GC_ISO1
  
  pure function GC_ISO2( a0, b0, c0, d0, e0, f0, g0, TEMP ) result( k )
    use types_mod
    ! Used to compute the rate for these reactions:
    !    ISOP + OH = 0.3MCO3 + 0.3MGLY + 0.3CH2O
    !              + 0.15HPALD3 + 0.25HPALD1 + 0.4HO2
    !              + 0.6CO + 1.5OH + 0.3HPETHNL + LISOPOH
    !    ISOP + OH = 0.3CH2O + 0.15HPALD4 + 0.25HPALD2
    !              + 1.5OH + 0.9CO + 0.7HO2 + 0.3MGLY
    !              + 0.3ATOOH + LISOPOH
    !
    real(kind=DP), intent(in) :: a0, b0, c0, d0, e0, f0, g0, TEMP
    real(kind=DP) :: k0, k1, k2, k
    !
    k0 = d0 * EXP( e0 / TEMP ) * EXP( 1.0E8_dp / TEMP**3 )
    k1 = f0 * EXP( g0 / TEMP )
    k2 = c0 * k0 / ( k0 + k1 )
    k  = a0 * EXP( b0 / TEMP ) * k2
  end function GC_ISO2
  
  pure function GC_NIT( a0, b0, c0, n, x0, y0, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Used to compute the rate for these reactions:
    !    IHOO1    + NO = IHN2
    !    IHOO4    + NO = IHN4
    !    IHPOO1   + NO = IHTN
    !    IHPOO2   + NO = IHTN
    !    IHPOO2   + NO = IHTN
    !    IEPOXAOO + NO = IHTN
    !    IEPOXBOO + NO = IHTN
    !    IHCOO    + NO = IHTN
    !    ISOPNOO1 + NO = IDN
    !    ISOPNOO2 + NO = IDN
    !    IDHNDOO1 + NO = IDN
    !    IDHNDOO2 + NO = IDN
    !    INO2B    + NO = IDN
    !    INO2D    + NO = IDN
    !    IHPNBOO  + NO = IDN
    !    IHPNDOO  + NO = IDN
    !    MVK0HOO  + NO = 0.438MVKN
    !    MCROHOO  + NO = MCRHN
    !
    real(kind=DP), intent(in) :: a0, b0, c0, n,  x0, y0, TEMP, NUMDEN
    real(kind=DP) :: k0, k1, k2, k3, k4, k
    !
    k0 = 2.0E-22_dp * EXP( n )
    k1 = 4.3E-1_dp * ( TEMP / 298.0_dp )**(-8)
    k0 = k0 * NUMDEN
    k1 = k0 / k1
    k2 = ( k0 / ( 1.0_dp + k1 ) )                                            &
       * 4.1E-1_dp**( 1.0_dp / ( 1.0_dp + ( LOG10(k1) )**2) )
    k3 = k2 / ( k2 + c0 )
    k4 = A0 * ( x0 - TEMP*y0 )
    k  = k4 * EXP( b0 / TEMP ) * k3
    k  = MAX( k, 0.0_dp )
  end function GC_NIT
  
  pure function GC_OHCO_a( a0, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Reaction rate for:
    !    OH + CO = HO2 + CO2 (cf. JPL 15-10)
    !
    ! For this reaction, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0 * EXP(c0/T)
    ! because b0 = c0 = 0.  Therefore we can skip computing these
    ! terms.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, TEMP, NUMDEN
    !
    real(kind=DP) :: klo1,   klo2,   khi1,  khi2
    real(kind=DP) :: xyrat1, xyrat2, blog1, blog2,   fexp1
    real(kind=DP) :: fexp2,  kco1,   kco2, k, K300_OVER_TEMP
    !
    K300_OVER_TEMP = 300.0_DP / TEMP
    klo1   = 5.9E-33_dp * K300_OVER_TEMP
    khi1   = 1.1E-12_dp * K300_OVER_TEMP**(-1.3_dp)
    xyrat1 = klo1 * NUMDEN / khi1
    blog1  = LOG10( xyrat1 )
    fexp1  = 1.0_dp / ( 1.0_dp + blog1*blog1 )
    kco1   = klo1 * NUMDEN * 0.6_dp**fexp1 / ( 1.0_dp + xyrat1 )
    klo2   = 1.5E-13_dp
    khi2   = 2.1E+09_dp * K300_OVER_TEMP**(-6.1_dp)
    xyrat2 = klo2 * NUMDEN / khi2
    blog2  = LOG10( xyrat2 )
    fexp2  = 1.0_dp / ( 1.0_dp + blog2*blog2 )
    kco2   = klo2 * 0.6_dp**fexp2 / ( 1.0_dp + xyrat2 )
    k      = kco1 + kco2
  end function GC_OHCO_a
  
  pure function GC_OHHNO3_acacac( a0, c0, a1, c1, a2, c2, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Used to compute the rate for these reactions:
    !    HNO3  + OH = H2O + NO3
    !    HONIT + OH = NO3 + HAC
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0
    !    (300/T)**b1
    !    (300/T)**b2
    ! Because b0 = b1 = b2 = 0.  Therefore we can skip computing
    ! these terms.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, a1, c1, a2, c2, TEMP, NUMDEN
    real(kind=DP) :: k0, k1, k2, k
    !
    ! ---  OH + HNO3:   K = K0 + K3[M] / (1 + K3[M]/K2)  ------
    k0   = a0 * EXP( c0 / TEMP )
    k1   = a1 * EXP( c1 / TEMP )
    k2   = NUMDEN * ( a2 * EXP( c2 / TEMP ) )
    k = k0 + k2 / ( 1.0_dp + k2/k1 )
  end function GC_OHHNO3_acacac
  
  pure function GC_PAN_abab( a0, b0, a1, b1, cf, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Used to compute the rate for these reactions:
    !    MACR1OO + NO2 = MPAN
    !    MACRNO2 + NO2 = MPAN + NO2
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    EXP(b0/T)
    !    EXP(b1/T)
    ! because b0 = b1 = 0.  Therefore we can skip computing these
    ! terms.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    ! Sept 27 2012: Added GC_PAN_abab per Kelvin Bates' requirements
    !               for aromatic chem.
    real(kind=DP), intent(in) :: a0, b0, a1, b1, cf, TEMP, NUMDEN
    real(kind=DP) :: k0, k1, kr, nc, f,  k
    !
    k0 = a0 * EXP( b0 / TEMP )
    k1 = a1 * EXP( b1 / TEMP )
    k0 = k0 * NUMDEN
    kr = k0 / k1
    nc = 0.75_dp - 1.27_dp * ( LOG10( cf ) )
    f  = 10.0_dp**( LOG10( cf ) / ( 1.0_dp + ( LOG10( kr ) / nc )**2 ) )
    k  = k0 * k1 * f / ( k0 + k1 )
  end function GC_PAN_abab
  
  pure function GC_PAN_acac( a0, c0, a1, c1, cf, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Used to compute the rate for these reactions:
    !    MACR1OO + NO2 = MPAN
    !    MACRNO2 + NO2 = MPAN + NO2
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    EXP(b0/T)
    !    EXP(b1/T)
    ! because b0 = b1 = 0.  Therefore we can skip computing these
    ! terms.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, a1, c1, cf, TEMP, NUMDEN
    real(kind=DP) :: k0, k1, kr, nc, f,  k, TEMP_OVER_K300
    !
    TEMP_OVER_K300 = TEMP / 300.0_DP
    k0 = a0 * TEMP_OVER_K300**c0
    k1 = a1 * TEMP_OVER_K300**c1
    k0 = k0 * NUMDEN
    kr = k0 / k1
    nc = 0.75_dp - 1.27_dp * ( LOG10( cf ) )
    f  = 10.0_dp**( LOG10( cf ) / ( 1.0_dp + ( LOG10( kr ) / nc )**2 ) )
    k  = k0 * k1 * f / ( k0 + k1 )
  end function GC_PAN_acac
  
  pure function GC_RO2HO2_aca( a0, c0, a1, TEMP ) result( k )
    use types_mod
    ! Carbon Dependence of RO2+HO2, used in these reactions:
    !    A3O2 + HO2 = RA3P
    !    PO2  + HO2 = PP
    !    KO2  + HO2 = 0.150OH + 0.150ALD2 + 0.150MCO3 + 0.850ATOOH
    !    B3O2 + HO2 = RB3P
    !    PRN1 + HO2 = PRPN
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0
    !    (300/T)**b1 * EXP(c1/T)
    ! Because b0 = b1 = c1 = 0.  Therefore we can skip computing these
    ! terms.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, a1, TEMP
    real(kind=DP) :: k
    !
    k = a0 * EXP( c0 / TEMP )
    k = k * ( 1.0_dp - EXP( -0.245_dp * a1 ) )
  end function GC_RO2HO2_aca
  
  pure function GC_RO2NO_A1_ac( a0, c0, TEMP ) result( k )
    use types_mod
    ! Reaction rate for the "A" branch of these RO2 + NO reactions:
    !    MO2  + NO = MENO3
    ! in which the "a1" parameter equals exactly 1.
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0
    !    (300/T)**b1 * EXP(c1/T)
    ! because b0 = b1 = c1 = 0.  Therefore we can skip computing
    ! these terms.  This avoids excess CPU cycles. (bmy, 1/4/20)
    !
    ! Special treatment for methyl nitrate based on observations
    ! as Carter and Atkinson formulation does not apply to C1.
    ! Value based on upper limit of Flocke et al. 1998 as applied
    ! in Fisher et al. 2018
    !
    real(kind=DP), intent(in) :: a0, c0, TEMP
    real(kind=DP) :: k
    !
    k  = a0 * EXP( c0 / TEMP ) * 3.0e-4_dp
  end function GC_RO2NO_A1_ac
  
  pure function GC_RO2NO_A2_aca( a0, c0, a1, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Reaction rate for the "A" branch of these RO2 + NO reactions,
    !    ETO2 + NO = ETNO3
    !    A3O2 + NO = NPRNO3
    !    R4O2 + NO = R4N2
    !    B3O2 + NO = IPRNO3
    ! in which the "a1" parameter is greater than 1.0.
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0
    !    (300/T)**b1 * EXP(c1/T)
    ! because b0 = b1 = c1 = 0.  Therefore we can skip computing
    ! these terms.  This avoids excess CPU cycles. (bmy, 1/4/20)
    !
    real(kind=DP), intent(in) :: a0,  c0,   a1, TEMP, NUMDEN
    real(kind=DP) :: k0,  k, yyyn, xxyn
    real(kind=DP) :: aaa, rarb, zzyn, fyrno3
    !
    k0     = a0 * EXP( c0 / TEMP )
    xxyn   = 1.94e-22_dp * EXP( 0.97_dp * a1 ) * NUMDEN
    yyyn   = 0.826_dp * ( ( 300.0_dp / TEMP )**8.1_dp )
    aaa    = LOG10( xxyn / yyyn )
    zzyn   = ( 1.0_dp / ( 1.0_dp + ( aaa *  aaa  ) ) )
    rarb   = ( xxyn   / ( 1.0_dp + ( xxyn / yyyn ) ) ) * ( 0.411_dp**zzyn )
    fyrno3 = ( rarb   / ( 1.0_dp +   rarb          ) )
    k      = k0 * fyrno3
  end function GC_RO2NO_A2_aca
  
  pure function GC_RO2NO_B1_ac( a0, c0, TEMP ) result( k )
    use types_mod
    ! Reaction rate for the "B" branch of these RO2 + NO reactions:
    !    MO2 + NO = CH2O + NO2 + HO2
    ! in which the "a1" parameter equals exactly 1.
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0
    !    (300/T)**b1 * EXP(c1/T)
    ! because b0 = c0 = c1 = 0.  Therefore we can skip computing
    ! these terms.  This avoids excess CPU cycles. (bmy, 1/4/20)
    !
    real(kind=DP), intent(in) :: a0, c0, TEMP
    real(kind=DP), PARAMETER  :: one_minus_fyrno3 = 1.0_dp - 3.0e-4_dp
    real(kind=DP) :: k
    !
    k = a0 * EXP( c0 / TEMP ) * one_minus_fyrno3
  end function GC_RO2NO_B1_ac
  
  pure function GC_RO2NO_B2_aca( a0, c0, a1, TEMP, NUMDEN ) result( k )
    use types_mod
    ! Reaction rate for the "B" branch of these RO2 + NO reactions:
    !    ETO2 + NO = NO2 +     HO2 + ...
    !    A3O2 + NO = NO2 +     HO2 + ...
    !    R4O2 + NO = NO2 + 0.27HO2 + ...
    !    B3O2 + NO = NO2 +     HO2 + ...
    ! in which the "a1" parameter is greater than 1.0.
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0
    !    (300/T)**b1 * EXP(c1/T)
    ! because b0 = c0 = c1 = 0.  Therefore we can skip computing
    ! these terms.  This avoids excess CPU cycles. (bmy, 1/4/20)
    !
    ! Use this function when a1 input argument is greater than 1.0.
    ! This avoids IF statements, which saves CPU cycles (bmy, 1/4/20)
    !
    real(kind=DP), intent(in) :: a0,  c0,   a1, TEMP, NUMDEN
    real(kind=DP) :: k0,  k, yyyn, xxyn, K300_OVER_TEMP
    real(kind=DP) :: aaa, rarb, zzyn, fyrno3
    !
    K300_OVER_TEMP = 300.0_DP / TEMP
    k0     = a0 * EXP( c0 / TEMP )
    xxyn   = 1.94e-22_dp * EXP(  0.97_dp * a1 ) * NUMDEN
    yyyn   = 0.826_dp * ( K300_OVER_TEMP**8.1_dp )
    aaa    = LOG10( xxyn / yyyn )
    zzyn   = ( 1.0_dp / ( 1.0_dp + ( aaa  * aaa  ) ) )
    rarb   = ( xxyn   / ( 1.0_dp + ( xxyn / yyyn ) ) ) * ( 0.411_dp**zzyn )
    fyrno3 = ( rarb   / ( 1.0_dp +   rarb          ) )
    k      = k0 * ( 1.0_dp - fyrno3 )
  end function GC_RO2NO_B2_aca
  
  pure function GC_TBRANCH_1_acac( a0, c0, a1, c1, TEMP ) result( k )
    use types_mod
    ! Temperature Dependent Branching Ratio, used for reactions:
    !    MO2 + MO2 = CH2O  + MOH + O2
    !    MO2 + MO2 = 2CH2O + 2HO2
    !
    ! For these reactions, these Arrhenius law terms evaluate to 1:
    !    (300/T)**b0
    !    (300/T)**b1
    ! because b0 = b1 = 0.  Therefore we can skip computing these
    ! terms.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, a1, c1, TEMP
    real(kind=DP) :: k0, k1, k
    !
    k0 = a0 * EXP( c0 / TEMP )
    k1 = a1 * EXP( c1 / TEMP )
    k  = k0 / ( 1.0_dp + k1 )
  end function GC_TBRANCH_1_acac
  
  pure function GC_TBRANCH_2_acabc( a0, c0, a1, b1, c1, TEMP ) result( k )
    use types_mod
    ! Temperature Dependent Branching Ratio, used for reactions:
    !    C3H8 + OH = B3O2
    !    C3H8 + OH = A3O2
    !
    ! For these reactions, this Arrhenius law term evaluates to 1:
    !    (300/T)**b0
    ! because b0 = 0.  Therefore we can skip computing this
    ! term.  This avoids excess CPU cycles. (bmy, 12/18/20)
    !
    real(kind=DP), intent(in) :: a0, c0, a1, b1, c1, TEMP
    real(kind=DP) :: k0, k1, k, K300_OVER_TEMP
    !
    K300_OVER_TEMP = 300.0_DP / TEMP
    k0 = a0 * EXP( c0 / TEMP )
    k1 = a1 * EXP( c1 / TEMP ) * K300_OVER_TEMP**b1
    k  = k0 / ( 1.0_dp + k1 )
  end function GC_TBRANCH_2_acabc
  
  pure function TUNPLUS_abcde( a0, b0, c0, d0, e0, TEMP ) result( k )
    use types_mod
    ! Used to compute the rate for these reactions:
    !    IHOO1 = 1.5OH + ...
    !    IHOO4 = 1.5OH + ...
    !
    real(kind=DP), intent(in) :: a0, b0, c0, d0, e0, TEMP
    real(kind=DP) :: k
    !
    k = a0 * ( d0 + ( TEMP * e0 ) )
    k = k * EXP( b0 / TEMP ) * EXP( c0 / TEMP**3 )
    k = MAX( k, 0.0_dp )
  end function TUNPLUS_abcde

  FUNCTION GCJPLAC_ababac( a1, b1, a2, b2, a3, c3, fv, TEMP, NUMDEN ) RESULT( k )
    use types_mod
    ! Rate coefficient for activation reactions competing with a 
    ! termolecular association pathway
    ! a1, b1 are the Arrhenius parameters for the lower-limit rate.
    ! a2, b2 are the Arrhenius parameters for the upper-limit rate.
    ! a3, c3 are Arrhenius parameters for the activation path
    ! fv     is the falloff curve parameter, usually = 0.6.
    !
    ! Used to compute the rate for these reactions:
    !    NO2  + O  = O2 + NO
    !    HNO3 + OH = NO3 + H2O
    !    CO   + OH = HO2 + CO2
    !
    real(kind=DP), intent(in) :: a1, b1, a2, b2, a3, c3, fv, TEMP, NUMDEN
    real(kind=DP)             :: rlow, rhigh, xyrat, blog, K300_OVER_TEMP
    real(kind=DP)             :: fexp, k1, k2, k
    !
    K300_OVER_TEMP = 300.0_DP / TEMP
    rlow  = a1 * ( K300_OVER_TEMP**b1 ) * NUMDEN
    rhigh = a2 * ( K300_OVER_TEMP**b2 )
    xyrat = rlow / rhigh
    blog  = LOG10( xyrat )
    fexp  = 1.0_dp / ( 1.0_dp + ( blog * blog ) )
    k1    = rlow * ( fv**fexp ) / ( 1.0_dp + xyrat )
    k2    = a3 * EXP( c3 / TEMP )
    k     = k2 * (1.0_dp - (k1 / rhigh) )
  END FUNCTION GCJPLAC_ababac

end module custom_functions_mod