function varargout = matchMASC(datastream,vars,loc,avg_time,settling_time,masc_datetime)

% Co-locates masc_particles in time with other ARM data
%
% VARARGOUT = MATCHMASC(DATASTREAM,VARS,LOC,AVG_TIME,SETTLING_TIME,MASC_DATETIME)
% Given a set of dates and times corresponding to observed MASC
% measurements (MASC_DATETIME), as well as the desired ARM DATASTREAM,
% variables of interest (VARS), ARM location (LOC), averaging time
% (AVG_TIME, in minutes), and SETTLING TIME (in minutes), this function 
% co-locates MASC particle measurements with other co-located ARM 
% measurements. SETTLING TIME is the time difference between those 
% measurements associated with clouds over the site and MASC particle 
% measurements at the surface.

% ---------------------------------
% Kyle E. Fitch, Maj, USAF (AFIT/CI)
% University of Utah
% Salt Lake City, Utah
% kyle.fitch@us.af.mil
% ---------------------------------

% Dependencies:
% 	uploaddirs_ARM3.m

% % example input:
% masc_datetime = [oli_year,oli_month,oli_day,oli_hour,oli_min,oli_sec];
% datastream = 'met';
% vars = {'wdir_vec_mean'};
% loc = 'oli';
% avg_time = 10; % time period over which to average ARM measurements (min)
% settling_time = 0; % account for hydrometeor settling time (min)
% datastream = 'met';
% vars={'temp_mean','wspd_arith_mean','atmos_pressure'};
% loc='oli';avg_time=10;settling_time=0; masc_datetime=double(masc_datetime);
% datastream = 'mwrret2turn'; 
% vars = {'phys_lwp','phys_lwp_uncertainty','tbsky_calc'}; 
% loc = 'oli'; avg_time = 10; settling_time = 10;

% make sure datetime is correct variable type
masc_datetime = double(masc_datetime);


%% constants
datenum_min = 1/1440; % 1 minute in datenum increments

%% set file path and data level based on requested datastream
if strcmp(datastream,'met') && strcmp(loc,'oli')
    ARM_data_loc = ['/uufs/chpc.utah.edu/common/home/garrett-group3/',...
        'fitch/Data/ARM/MET/OLI'];
    data_lvl = 'b1';
    
elseif strcmp(datastream,'mwr') && strcmp(loc,'oli')
    ARM_data_loc = ['/uufs/chpc.utah.edu/common/home/garrett-group3/',...
        'fitch/Data/ARM/MWR-LWP'];
    data_lvl = 'b1';
    
elseif strcmp(datastream,'skyrad') && strcmp(loc,'oli')
    ARM_data_loc = ['/uufs/chpc.utah.edu/common/home/garrett-group3/',...
        'fitch/Data/ARM/SKYRAD-Tcb'];
    data_lvl = 'b1';
    
elseif strcmp(datastream,'mwrret2turn') && strcmp(loc,'oli')
    ARM_data_loc = ['/uufs/chpc.utah.edu/common/home/garrett-group3/',...
        'fitch/Data/ARM/LWP_corr_Maahn'];
    data_lvl = 'c1';    
elseif strcmp(datastream,'kollias') && strcmp(loc,'oli')
    ARM_data_loc = ['/uufs/chpc.utah.edu/common/home/garrett-group3/',...
        'fitch/Data/ARM/Kollias'];
    data_lvl = 'c0';
elseif strcmp(datastream,'dlprof') && strcmp(loc,'oli')
    ARM_data_loc = ['/uufs/chpc.utah.edu/common/home/garrett-group3/',...
        'fitch/Data/ARM/dlprof'];
    data_lvl = 'c1';
end

%% find ARM data
cd(ARM_data_loc)

file_list = dir([loc,'*']);
file_names = {file_list.name};
[new_file_list,new_fn_list] = uploaddirs_ARM3(ARM_data_loc,file_names,...
    masc_datetime(1,1:4),masc_datetime(end,1:4)); %outputs: directory list

ARM_datenum = [];

for n = 1:length(vars)
    eval([vars{n} '= [];'])
end

for m = 1:length(new_file_list)
    fn = new_fn_list{m};
    mydatestr = extractBetween(fn,[data_lvl,'.'],'.');
    
    % extract date from filename
    ARM_year = str2double(mydatestr{:}(1:4));
    ARM_month = str2double(mydatestr{:}(5:6));
    ARM_day = str2double(mydatestr{:}(7:8));
    
    % if using Turner-corrected MWR data, get data for QC check for wet sensor
    if strcmp(datastream,'mwrret2turn')
        ARM_time = ncread(fn,'time_offset');
        freq = ncread(fn,'freq');
        ind_31p4 = find(abs(freq-31.4000)<0.01);
    else
        ARM_time = ncread(fn,'time'); % time in seconds since midnight (s)
    end
    
    % datetime matrices for ARM measurements
    ARM_datevec_base = [ARM_year,ARM_month,ARM_day,0,0,0];
    ARM_datetime = datetime(ARM_datevec_base) + ARM_time./(3600*24);
    ARM_datenum = [ARM_datenum;datenum(ARM_datetime)];
    
    % auto assign variable names using 'eval' function
    for n = 1:length(vars)
        % if using Turner-corrected MWR data, do QC check for wet sensor
        if strcmp(vars{n},'tbsky_calc')
            tmp = ncread(fn,'tbsky_calc');
            tbsky_calc = [tbsky_calc;tmp(ind_31p4,:)'];
        else
            eval([vars{n} '= [',vars{n} ';ncread(fn,vars{n})];'])
        end
    end
end

% don't use wet mwr data when calculating mean lwp
if strcmp(datastream,'mwrret2turn')
    phys_lwp(tbsky_calc>=100) = nan;
    phys_lwp_uncertainty(tbsky_calc>=100) = nan;
    
    % any negative LWP values should be set to 0
    phys_lwp(phys_lwp<0) = 0;
end

%% run the loop to match times

for n = 1:length(vars)
    eval([vars{n},'2','= nan(size(masc_datetime,1),1);'])
end

disp('started matching loop');
for masc_var = 1:size(masc_datetime,1)
    if masc_var == round(size(masc_datetime,1)/100)
        disp('matching 1% complete')
    elseif masc_var == round(size(masc_datetime,1)/10)
        disp('matching 10% complete');
    elseif masc_var == round(size(masc_datetime,1)/4)
        disp('matching 25% complete');
    elseif masc_var == round(size(masc_datetime,1)/2)
        disp('matching 50% complete');
    elseif masc_var == round(size(masc_datetime,1)*(3/4))
        disp('matching 75% complete');
    elseif masc_var == round(size(masc_datetime,1)*(9/10))
        disp('matching 90% complete');        
    end
   
    % find MASC datetime period
    masc_datevec = masc_datetime(masc_var,:);
    masc_datenum = datenum(masc_datevec)-settling_time.*datenum_min;
    
    % match & average
    if strcmp(vars{n},'wdir_vec_mean') || strcmp(vars{n},'wspd_arith_mean')
        datenum_diff = masc_datenum-ARM_datenum;
        % sfc wind data calculated for time period leading up to obs
        avg_ind = find(datenum_diff <= avg_time*datenum_min & ...
            datenum_diff > 0);
    % if not sfc wind, find indices for averaging period spanning obs time    
    else
        datenum_diff = abs(masc_datenum-ARM_datenum);
        avg_ind = find(datenum_diff <= (avg_time/2)*datenum_min);
    end
    
    % do averaging (or find mode for wind direction)
    if ~isempty(avg_ind)
        for n = 1:length(vars)
            if strcmp(vars{n},'wdir_vec_mean')
                eval([vars{n},'2(masc_var)','= mode(',vars{n},'(avg_ind));'])
            else
                eval([vars{n},'2(masc_var)','= nanmean(',vars{n},'(avg_ind));'])
            end
        end
    end
end
disp('matching complete');

% out variables
varargout = {};
for n = 1:length(vars)
    eval(['varargout{n} =', vars{n},'2;'])
end