;; $ID$ 
;+
;############### TES Level 2 IDL Code ###############################
;;
;;                      Tropospheric Emission
;;                          Spectrometer
;;                             (TES)
;;
;;                       Level 2 Subsystem
;;
;;                         Copyright 2002,
;;              California Institute of Technology.
;;                       ALL RIGHTS RESERVED.
;;           U.S. Government Sponsorship acknowledged.
;;
;;
;; ROUTINE NAME
;;    sqrt_matrix
;;
;; FILE NAME
;;    sqrt_matrix.pro
;;
;; ABSTRACT
;;    Calculates the "square root" of a *symmetric* square matrix by use of the
;;    SVD.  The square root can be written as
;;    A = S S^T .  
;;
;;   Note that this routine assumes that the square matrix is
;;   symmetric. 
;;       
;; AUTHOR
;;    Kevin Bowman, JPL, July, 2002
;;
;;
;; REVISION HISTORY
;;    Kevin Bowman, JPL, July, 2002, Initial version
;;    Kevin Bowman, JPL, March, 2003, Switched SVD routine to LA_SVD
;;    for IDL 5.6.
;;
;; RETURN
;;    sqmatrix (double matrix): Square root of the input matrix
;;    
;;
;; INPUT 
;;    matrix (double matrix): The input matrix to be factored. 
;;
;;
;; IDL ROUTINES
;;
;;    Requires la_svd, which is only available on idl5.6 or higher. 
;;  
;;
;;####################################################################
;-
FUNCTION sqrt_matrix,matrix
tol = 1e-6
dblerr = 1e-10
dim = size(matrix,/dimensions)
IF dim[0] NE dim[1] THEN BEGIN
   PRINT,'Matrix must be square'
   STOP
ENDIF 

; Calculate SVD of input matrix
la_svd,matrix,svmatrix,wmatrix,vmatrix

num = n_elements(svmatrix)

; VERIFY THAT LEFT AND RIGHT SINGULAR VECTORS ARE EQUAL
svneg = dblarr(num)+1.0
FOR ii = 0,num-1 DO BEGIN
    nerr = norm(wmatrix[ii,*]-vmatrix[ii,*])
    IF nerr GT dblerr THEN svneg[ii] = -1D
ENDFOR 
negfind = where(svneg eq -1)
;IF negfind[0] NE -1 THEN PRINT,'Warning: Matrix has negative singular values and/or is non-symmetric'

; Scale Singular matrix with sqrt of singular values
dsvmatrix = dblarr(dim[0],dim[1])
    dsvmatrix[indgen(dim[0]),indgen(dim[0])] = sqrt(svmatrix)
sqmatrix = vmatrix##dsvmatrix

error = norm(sqmatrix##transpose(sqmatrix)-matrix)/norm(matrix)
;IF (error GE tol) THEN PRINT,'Warning: ||LL^T-A||/||A|| > tol, tol: ', tol,' error:', error
RETURN,sqmatrix
END
