;**********************************************
;  This code performs the nd and precip retrieval for ENA
;  
;  The procedures herein are documented in the following publications
;  
;  Mace, G.: Deriving cloud droplet number concentration from surface based remote sensors with an emphasis on lidar measurements, Atmos. Meas. Tech., 17, 3679-3695, https://doi.org/10.5194/amt-17-3679, 2024.
;  
;  Mace, G. G., Sally Benson, Elizabeth Sterner, et al. The Association Between Cloud Droplet Number Over the Summer Southern Ocean and Air Mass History, Journal of Geophysical Research, Atmospheres, 129, e2023jd040673, https://doi.org/10.1029/2023JD040673.
;  
;  Mace, G. G., Benson, S., Gombert, P., and Smallwood, T.: Impact on Cloud Properties of Reduced-Sulphur Shipping Fuel in the Eastern North Atlantic, EGUsphere [preprint], https://doi.org/10.5194/egusphere-2025-2075, 2025.
;**********************************************

pro ena_nd_retrieval_clean

;  Program
program='ENA'
eta_flag=0  ; 0 means to use the calculated eta.  Otherwise use 0.97 - a value typical of the period after the fix on 1/12/2018

;  Number of profiles in the running average for the nd retrieval

num_to_avg=10.0  ;marcus 30*10=300sec=5min

;top path to ena data files
path_prefix='/Users/u0029340/Documents/data/ena/'

;  The output file directory 
outdir=path_prefix+'temp' ;'cloud_precip_0p2/2023_10p0_75/'

;  Marcus data directory.  Change year to process separate years
fdir=path_prefix+'ena.avg.30sec.15m/2023/'

;  Missing value
missing=-8888

; lidar cloudbase threshold
cloud_base_threshold=10.0


;  Run through this list of dates specified in the next line
julian_day_str=timegen(start=julday(8,21,2023,0,0,0),final=julday(8,22,2023,0,0,0),$
  units='days',step_size=1)
  
caldat,julian_day_str,mm,dd,yy,hh,mi,ss
date_str=string(yy,format='(I4)')+string(mm,format='(I02)')+string(dd,format='(I02)')

jday1=julday(1,1,1970,0,0,0)
  
; hours to process
start_hr=1. & end_hr=23.  ; 20180102


;  Loop through list of dates
for ddd=0,n_elements(date_str)-1 do begin
  plot_flag=0
  
  ;  are all the data files present?
  outname='nd_retrieval.ena.'+date_str[ddd]+'.avg.30sec.15m.cdf'
  result=file_search(fdir+'ena_data.'+date_str[ddd]+'.avg.30sec.15m.cdf', count=count_files_30s)
  ;;;; get the ena ccn and surface wind data
  ; paths to additional data files
  sfc_met_dir='/Users/u0029340/Documents/data/ena/enaaosmetC1.a1/'
  ccn_dir='/Users/u0029340/Documents/data/ena/enaaosccn1colavgC1.b1/'
  ccn_dir='/Users/u0029340/Documents/data/ena/enaaosccn2colaavgC1.b1/'
  ; check to see that the data files exist
  smet_files=file_search(sfc_met_dir+'enaaosmetC1.a1.'+date_str[ddd]+'*.*', count=count_smet)
  ccn1_files=file_search(ccn_dir+'enaaosccn1colavgC1.b1.'+date_str[ddd]+'*.nc', count=count_ccn) ; pre 2020 ccn
  ccn1_files=file_search(ccn_dir+'enaaosccn2colaavgC1.b1.'+date_str[ddd]+'*.nc', count=count_ccn) ; post 2020 ccn

if count_files_30s gt 0 and count_smet gt 0 and count_ccn gt 0 then begin

  ;  The output file name for the data prepped for retrieval
  outname='nd_retrieval.ena.'+date_str[ddd]+'.avg.30sec.15m.cdf'
  result=file_search(fdir+'ena_data.'+date_str[ddd]+'.avg.30sec.15m.cdf', count=count_files)
  if count_files ge 1 then yes_do=1
  if count_files le 0 then yes_do=0

  if yes_do eq 1 then begin

    ;  Read in marcus data on 30s 15m grid
    fname='ena_data.'+date_str[ddd]+'.avg.30sec.15m.cdf'
    file_exist=file_test(fdir+fname)
    if file_exist eq 1 then begin
      fid=ncdf_open(fdir+fname)
      vid=ncdf_varid(fid,'julian_day') & ncdf_varget,fid,vid,julian_day
      vid=ncdf_varid(fid,'lat') & ncdf_varget,fid,vid,lat
      vid=ncdf_varid(fid,'lon') & ncdf_varget,fid,vid,lon
      vid=ncdf_varid(fid,'solar_zenith') & ncdf_varget,fid,vid,solar_zenith
      vid=ncdf_varid(fid,'lidar_bscat_copol') & ncdf_varget,fid,vid,lidar_bscat_copol ;NRB cts/(us*km2)
      vid=ncdf_varid(fid,'lidar_bscat_crospol') & ncdf_varget,fid,vid,lidar_bscat_crospol ;NRB cts/(us*km2)
      vid=ncdf_varid(fid,'cloud_base_ceil') & ncdf_varget,fid,vid,cloud_base_ceil ;km
      vid=ncdf_varid(fid,'sonde_dt') & ncdf_varget,fid,vid,sonde_dt ;minutes
      vid=ncdf_varid(fid,'temp') & ncdf_varget,fid,vid,temp  ;Celsius
      vid=ncdf_varid(fid,'pres') & ncdf_varget,fid,vid,pres  ;mb
      vid=ncdf_varid(fid,'rh') & ncdf_varget,fid,vid,rh      ;%
      vid=ncdf_varid(fid,'dp') & ncdf_varget,fid,vid,dpt     ;Celsius
      vid=ncdf_varid(fid,'liq_mwr') & ncdf_varget,fid,vid,liq_mwr ;cm  ;mwrlos
      vid=ncdf_varid(fid,'be_lwp') & ncdf_varget,fid,vid,be_lwp ;g/m2  ;mwrret
      if max(be_lwp) lt 0. and max(liq_mwr) gt 0. then begin
        be_lwp=liq_mwr
        i=where(be_lwp gt 0.)
        be_lwp[i]=be_lwp[i]*1.e3
      endif
      
      vid=ncdf_varid(fid,'cloud_mask_wacr') & ncdf_varget,fid,vid,cloud_mask_wacr
      vid=ncdf_varid(fid,'dbz_wacr') & ncdf_varget,fid,vid,dbz_wacr  ;dBZ
      vid=ncdf_varid(fid,'height') & ncdf_varget,fid,vid,height  ;km
      vid=ncdf_varid(fid,'bad_wacr_data') & ncdf_varget,fid,vid,bad_wacr_data  ;1=bad, 0=good
      vid=ncdf_varid(fid,'doppler_velocity_wacr') & ncdf_varget,fid,vid,doppler_velocity_wacr  ;1=bad, 0=good
      ncdf_close,fid
      caldat,julian_day,mm,dd,yy,hh,mi,ss
      lidar_hrfrac=fltarr(n_elements(hh))
      lidar_hrfrac=float(hh)+float(mi)/60.+float(ss)/3600.
      print,yy[0],mm[0],dd[0],hh[0],mi[0],ss[0],' radar-lidar data start'
      print,yy[-1],mm[-1],dd[-1],hh[-1],mi[-1],ss[-1],' radar-lidar data start'
      
    eta_flag=0
    ktop=min(where(height gt 4.))
    for k=0,ktop do begin
      factor=5.-((5./4.)*height[k]) ; test used for marcus
      lidar_bscat_copol[*,k]=lidar_bscat_copol[*,k]/factor  ; correct mpl data
    endfor
    
    start_j=min(where(lidar_hrfrac ge start_hr))
    end_j=max(where(lidar_hrfrac le end_hr))

      ;  data dimensions
      s=size(dbz_wacr,/dimensions)
      numtimes=s[0]
      numheights=s[1]

;;;; get the ena ccn and surface wind data
sfc_met_dir='/Users/u0029340/Documents/data/ena/enaaosmetC1.a1/'
ccn_dir='/Users/u0029340/Documents/data/ena/enaaosccn1colavgC1.b1/'
ccn_dir='/Users/u0029340/Documents/data/ena/enaaosccn2colaavgC1.b1/'

;;;;;;;;; get sfc met

smet_files=file_search(sfc_met_dir+'enaaosmetC1.a1.'+date_str[ddd]+'*.*', count=count)

for j=0,count-1 do begin

  cdfid=ncdf_open(smet_files[j])

  x_id=ncdf_varid(cdfid, 'base_time') & ncdf_varget,cdfid,x_id,base_time
  x_id=ncdf_varid(cdfid, 'time_offset') & ncdf_varget,cdfid,x_id,time_offset
  ;if n_elements(time_offset) gt 5000 then stop
  x_id=ncdf_varid(cdfid, 'wind_speed')
  if x_id gt 0. then begin
    ncdf_varget,cdfid,x_id,wind_speed1
    x_id=ncdf_varid(cdfid, 'wind_direction') & ncdf_varget,cdfid,x_id,wind_dir1
  endif else begin
    x_id=ncdf_varid(cdfid, 'WindDirection') & ncdf_varget,cdfid,x_id,wind_dir1
    x_id=ncdf_varid(cdfid, 'WindSpeed') & ncdf_varget,cdfid,x_id,wind_speed1
  endelse
  ncdf_close, cdfid

  ; subsample the met data
  for kk=0,n_elements(time_offset),25 do begin
    if kk lt n_elements(time_offset)-1 then begin
      if kk eq 0 then begin
        time_offset_temp=time_offset[kk]
        wind_dir_temp=wind_dir1[kk]
        wind_speed_temp=wind_speed1[kk]
      endif else begin
        time_offset_temp=[time_offset_temp,time_offset[kk]]
        wind_dir_temp=[wind_dir_temp,wind_dir1[kk]]
        wind_speed_temp=[wind_speed_temp,wind_speed1[kk]]
      endelse
    endif
  endfor
  time_offset=time_offset_temp
  wind_dir1=wind_dir_temp
  wind_speed1=wind_speed_temp

  julday1=jday1+((double(base_time)+time_offset)/86400.d)

  if j eq 0 then begin
    julday_smet=julday1
    wind_speed=wind_speed1
    wind_dir=wind_dir1
  endif else begin
    julday_smet=[julday_smet,julday1]
    wind_speed=[wind_speed, wind_speed1]
    wind_dir=[wind_dir, wind_dir1]
  endelse

endfor


;;;;; identify when winds are from a direction that avoids island influence
i=where(wind_dir gt 330. and wind_dir lt 360.)
wind_dir[i]=30.
i=where(wind_dir gt 220. and wind_dir lt 250.)
wind_dir[i]=30.
wind_conditions_vec=where(wind_speed gt 2.5 and wind_dir gt 0. and wind_dir lt 70., count_wind)
wind_filter=fltarr(n_elements(julday_smet))
if count_wind gt 0 then wind_filter[wind_conditions_vec]=1.

wf=wind_filter
for jj=30,n_elements(wind_filter)-31 do begin
  if total(wf[jj-30:jj+30]) gt 40 then begin
    wind_filter[jj]=1.
  endif else begin
    wind_filter[jj]=0.
  endelse
endfor
  

;;;;;; done sfc met
;;;; get ccn

ccn1_files=file_search(ccn_dir+'enaaosccn1colavgC1.b1.'+date_str[ddd]+'*.nc', count=count) ; pre 2020 ccn
                               ;'enaaosccn1colavgC1.b1.'
ccn1_files=file_search(ccn_dir+'enaaosccn2colaavgC1.b1.'+date_str[ddd]+'*.nc', count=count) ; post 2020 ccn

if n_elements(where(wind_filter gt 0)) gt 20. then begin
  
  caldat,julday_smet[where(wind_filter gt 0)], y,m,d,h,mi,s
  print, 'north wind times are from ',min(h), ' to ', max(h)
  
  ; put the wind filter into the same time grid as the cloud data.
  wind_filter_data=fltarr(n_elements(julian_day))
  for jj=0,n_elements(julian_day)-1 do begin
    i=where(abs(julian_day[jj]-julday_smet) eq min(abs(julian_day[jj]-julday_smet)))
    if abs(julian_day[jj]-julday_smet[i[0]]) lt 1./24. then begin
      wind_filter_data[jj]=wind_filter[i[0]]
    endif
  endfor
  wind_filter=wind_filter_data

for j=0,count-1 do begin
  cdfid=ncdf_open(ccn1_files[j])
  ;cdfid=ncdf_open(radar_path+radar_file,/nowrite)

  x_id=ncdf_varid(cdfid, 'base_time') & ncdf_varget,cdfid,x_id,base_time
  x_id=ncdf_varid(cdfid, 'time_offset') & ncdf_varget,cdfid,x_id,time_offset
  x_id=ncdf_varid(cdfid, 'supersaturation_calculated') & ncdf_varget,cdfid,x_id,ss_calc1
  x_id=ncdf_varid(cdfid, 'aerosol_number_concentration') & ncdf_varget,cdfid,x_id,cn1
  x_id=ncdf_varid(cdfid, 'N_CCN') & ncdf_varget,cdfid,x_id,N_CCN1
  x_id=ncdf_varid(cdfid, 'qc_N_CCN') & ncdf_varget,cdfid,x_id,qc_N_CCN1
  
  
  ncdf_close, cdfid

;  print, months[jj], n_elements(N_CCN1)

  julday1=jday1+((double(base_time)+time_offset)/86400.d)
  
  ccn_ss_regression, julday1, ss_calc1, N_CCN1,qc_N_CCN1, cn1,    0.2,   ccn_out, julday_out, cn_out, qc_out
ss_out=fltarr(n_elements(julday_out)) & ss_out[*]=0.3

  if j eq 0 then begin
    julday_ccn=julday_out
    ss_calc_ccn=ss_out
    cn_ccn=cn_out
    ccn_ccn=ccn_out
    qc_ccn_ccn=qc_out
  endif else begin
    julday_ccn=[julday_ccn,julday_out]
    ss_calc_ccn=[ss_calc_ccn,ss_out]
    cn_ccn=[cn_ccn,cn_out]
    ccn_ccn=[ccn_ccn,ccn_out]
    qc_ccn_ccn=[qc_ccn_ccn,qc_out]
  endelse

endfor  ; for j=0,count-1 do begin
  
  i=where(ss_calc_ccn gt 0.10 and ss_calc_ccn lt 0.35, count)
if count gt 0. then begin
  julday_ccn=julday_ccn[i]
  ccn=ccn_ccn[i]
  sdv=stddev(ccn)
  sdv_ccn=fltarr(n_elements(i))
  sdv_ccn[*]=sdv
endif else begin
  julday_ccn=-1.
  ccn=-1.
  sdv_ccn=-1
endelse

;; these are the variable names used below
ccn_0p2_med_filter=ccn
ccn_0p2_sdv_filter=sdv_ccn


      ;  Calculate a kelvin temperature array
      temp_k=temp
      r=where(temp_k ne missing,c)
      if c gt 0 then temp_k[r]=temp_k[r]+273.15 ;celsius->Kelvin

      ;  range gate in km 
      height_interval=(height[11]-height[10])  ;km

      ;*****************************
      ;  Classify the profiles
      ;  Definition of low cloud is ceil base lt 4km  
      ;  Only retrieving Nd for low clouds
      ;*****************************

      ;  Profiles with ceil cloud base > 0 and ceil cloud base < 4 km
      low_cloud_flag=make_array(numtimes,/int,value=0)  ;1=low cloud
      r=where(cloud_base_ceil gt 0.1 and cloud_base_ceil lt 4.0 and bad_wacr_data eq 0 and wind_filter eq 1. and be_lwp gt 0.,low_cloud_count)
      if low_cloud_count gt 0 then begin
        low_cloud_flag[r]=1
      endif
      
      if max(low_cloud_flag) eq 1 then begin
      
      ;  profiles with ceil cloud base = 0 or ceil cloud base > 10 km
      clear_data_flag=make_array(numtimes,/int,value=0) ;1=clear
      r=where(cloud_base_ceil eq 0 or cloud_base_ceil gt 10 and bad_wacr_data eq 0,clear_count)
      if clear_count gt 0 then begin
        clear_data_flag[r]=1
      endif

      ;  These are the variables we want to save from this section, written in the next section
      var_flag=['low_cloud_flag','clear_data_flag']
      name_flag=['Does profile have ceil base between 0.1km and 4.0km?',$
                 'Is profile clear between 0km and 10km?']
      unit_flag=['1=low cloud 0=no',$
                 '1=clear 0=cloudy']

      ;*************************************************
      ;  Lidar Ratio sc calculation
      ;*************************************************
      ;  Hu, Y. (2007), Depolarization ratio–effective lidar ratio relation:
      ;  Theoretical basis for space lidar cloud phase discrimination,
      ;  Geophys. Res. Lett., 34, L11812, doi:10.1029/2007GL029584.
      ;  Sc=single-scattering lidar ratio
      ;  eta=layer-effective multiple scattering factor
      ;  For water clouds, Sc has a near constant value very close to 19 sr
      ;  Sc is only calculated on low cloud profiles.
      ;*************************************************

      ;  For every profile, do an inital calculation of layer integrated values,
      ;  calculate an adjustment,and then recalculate the layer intergrated values.
      ;  The goal is to calculate eta
      
      ;  Signal thresholds for calculating eta
      ;  thresholds for caparicorn copol=0.01 crospol=0.0001 
      copol_threshold=0.1
      crospol_threshold=0.01

      ;  For plotting only
      adj_top1=fltarr(numtimes)  ;first sc adjustment region
      adj_base1=fltarr(numtimes)
      adj_top2=fltarr(numtimes)  ;second sc adjustment region
      adj_base2=fltarr(numtimes)
  
      ;  Variables used in calculation
      integ_tab=fltarr(numtimes) ;Integrated total attenuated backscatter
      integ_llab=fltarr(numtimes) ;Integrated copol
      integ_pab=fltarr(numtimes)  ;Integrated crospol
      integ_depol=make_array(/float,numtimes,value=missing) ;Integrated depol
      eta=make_array(/float,numtimes,value=missing) ;multiple scattering factor
      s_eff=make_array(/float,numtimes,value=missing) ;effective lidar ratio
      sc=make_array(/float,numtimes,value=missing) ;single scattering lidar ratio
      effective_radius=make_array(/float,numtimes,value=missing) ;effective radius from sc param
      ; this is the quantity needed to get a lidar ratio of 18.5.
      ccc_vector=make_array(/float,numtimes,value=missing) 
  
      ;  These arrays will hold the original lidar before adjustment
      lidar_bscat_copol_orig=lidar_bscat_copol
      lidar_bscat_crospol_orig=lidar_bscat_crospol
  
      ;  Loop through the profiles 
      ;for j=0,numtimes-1 do begin
       for j=start_j, end_j  do begin

        ;  If this is a low cloud defined as lidar cloud base < 4
        if low_cloud_flag[j] eq 1 then begin
      
          ;  Now loop through the heights of the profile
          for k=0,numheights-2 do begin
      
            ;  This height is between (2 range gates below ceil base) and 1km above (2 range gates below ceil base).
            if height[k] ge cloud_base_ceil[j]-2*height_interval and $
               height[k] lt (cloud_base_ceil[j]-2*height_interval)+1.0 then begin
          
              ;  The lidar signal has to meet a threshold for this calculation
              if lidar_bscat_copol[j,k] gt copol_threshold and lidar_bscat_crospol[j,k] gt crospol_threshold then begin
                ;  Hold these bases for plotting
                if adj_base1[j] eq 0 then adj_base1[j]=height[k]
                adj_top1[j]=height[k]
                ;  This is integrated total  
                integ_tab[j]=integ_tab[j]+((lidar_bscat_copol[j,k]+2.0*lidar_bscat_crospol[j,k])*height_interval)
                ;  This is integrated copol
                integ_llab[j]=integ_llab[j]+((lidar_bscat_copol[j,k])*height_interval)
                ;  This is integrated cross-pol
                integ_pab[j]=integ_pab[j]+((lidar_bscat_crospol[j,k])*height_interval)
                ;  This is layer integrated depolarization ratio
                integ_depol[j]=integ_pab[j]/integ_llab[j]
                ;  Hu 2007 equation 2
                if eta_flag eq 0 then begin
                  eta[j]=((1.-integ_depol[j])/(1.+integ_depol[j]))^2
                endif else begin
                  eta[j]=0.65
                endelse
                ;  Hu 2007 equation 3 for opaque layers
                s_eff[j]=1./(2.*integ_tab[j])
                ;  Hu 2007 equation 3
                sc[j]=s_eff[j]/eta[j]   
                ; This is a fit to re and lidar ratio S using in situ data.
                effective_radius[j]=169.72-(8.53*sc[j])  ;microns
              endif ;lidar signal is strong enough  
            endif  ;near the ceil base
          endfor ;end of loop through heights
          ;print,'initial calculation of layer integrated values',integ_tab[j],sc[j],effective_radius[j],eta[j]

          ;  Sc=single scattering lidar ratio
          if sc[j] gt 0. then begin
      
            ;  Calculate ccc, the adjustment factor
            ccc=1./(eta[j]*37.5*integ_tab[j]) 
            ccc_vector[j]=ccc
        
            ;  Apply the adjustment factor to the lidar profile
            lidar_bscat_copol[j,*]=lidar_bscat_copol[j,*]*ccc
            lidar_bscat_crospol[j,*]=lidar_bscat_crospol[j,*]*ccc
        
            ;  Reset integrated values to do adjusted calculation
            integ_tab[j]=0. & integ_llab[j]=0. & integ_pab[j]=0. 
        
            ;  Loop through the lidar heights
            for k=0,numheights-1 do begin
        
              ;  This height is between (2 range gates below ceil base) and 1km above (2 range gates below ceil base).
              if height[k] ge cloud_base_ceil[j]-2*height_interval and $
                 height[k] lt (cloud_base_ceil[j]-2*height_interval)+1.0 then begin
           
                ;  The lidar signal has to meet a threshold for this calculation
                if lidar_bscat_copol[j,k] gt copol_threshold*ccc and lidar_bscat_crospol[j,k] gt crospol_threshold*ccc then begin  
                  ;  Hold these bases for plotting
                  if adj_base2[j] eq 0 then adj_base2[j]=height[k]
                  adj_top2[j]=height[k]
                  ;  This is integrated total  ;it may need to be changed to what connor said
                  integ_tab[j]=integ_tab[j]+((lidar_bscat_copol[j,k]+2.0*lidar_bscat_crospol[j,k])*height_interval)
                  ;  This is integrated co-pol
                  integ_llab[j]=integ_llab[j]+((lidar_bscat_copol[j,k])*height_interval)
                  ;  This is integrated cross pol
                  integ_pab[j]=integ_pab[j]+((lidar_bscat_crospol[j,k])*height_interval)
                  ;  This is layer integrated depolarization ratio
                  integ_depol[j]=integ_pab[j]/integ_llab[j]
                  ;  Hu 2007 equation 2
                  if eta_flag eq 0 then begin
                    eta_new=((1.-integ_depol[j])/(1.+integ_depol[j]))^2
                    ;eta_new=0.6
                  endif else begin
                    eta_new=0.7
                  endelse
                  ;  Hu 2007 equation 3 for opaque layers
                  s_eff_new=1./(2.*integ_tab[j])
                  ;  Hu 2007 equation 3
                  sc_new=s_eff_new/eta_new   ;     (1./(2.*gamma_ss))*1.  ; multiplied by 0.5
                  ; this is a fit to re and lidar ratio S using in situ data.
                  effective_radius_new=169.72-(8.53*sc_new)  ;microns
                endif  ;lidar signal is strong enough
              endif  ;near the ceil base
            endfor  ;end of loop through heights
        
            ;@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
            ;  Remove the adjustment factor to the lidar profile, the values are now too small for other calculations
            ;  eta and integ_depol remain the same, only sc and integrated co/cros values change due to adjustment factor.
            lidar_bscat_copol[j,*]=lidar_bscat_copol[j,*]/ccc
            lidar_bscat_crospol[j,*]=lidar_bscat_crospol[j,*]/ccc
        
            eta[j]=eta_new
            s_eff[j]=s_eff_new
            sc[j]=sc_new   
            effective_radius[j]=169.72-(8.53*sc[j])  ;microns
            ;print,'second calculation of layer integrated values',integ_tab[j],sc[j],effective_radius[j],eta[j]
          endif  ;Sc lidar ratio calculated
        endif  ;if low cloud
      endfor  ;loop through profiles 
  
      ;  These are the variables we want to save from this section
      var_cal=[['integ_tab','integ_llab','integ_pab','integ_depol','eta','sc','effective_radius',$
        'ccc_vector'],var_flag]
      name_cal=[['mpl layer integrated total attenuated backscatter',$
                'mpl layer integrated copol attenuated backscatter',$
                'mpl layer integrated crosspol attenuated backscatter',$
                'mpl layer integrated depolarization',$
                'layer effective multiple scattering factor',$
                'single scattering lidar ratio',$
                'effective radius from sc parameterization',$
                'sc adjustment factor for attenuated backscatter profiles'],name_flag]
      unit_cal=[['Normalized Relative Backscatter (NRB) cts/(us*km)','NRB cts/(us*km)','NRB cts/(us*km)','unitless',$
                 'unitless','unitless','micron','unitless'],unit_flag]
            
      ;  Write these variables to a file
      fid=ncdf_create(outdir+outname,/clobber)
      num_did=ncdf_dimdef(fid,'numtimes',numtimes)
      julian_day_id=ncdf_vardef(fid,'julian_day',num_did,/double)
      for i=0,n_elements(var_cal)-1 do begin
        xstr=var_cal[i]+'_id=ncdf_vardef(fid,"'+var_cal[i]+'",num_did)'
        result=execute(xstr)
        xstr='ncdf_attput,fid,'+var_cal[i]+'_id,"long_name","'+name_cal[i]+'"'
        result=execute(xstr)
        xstr='ncdf_attput,fid,'+var_cal[i]+'_id,"units","'+unit_cal[i]+'"'
        result=execute(xstr)
      endfor
      ncdf_control,fid,/endef
      ncdf_varput,fid,julian_day_id,julian_day
      for i=0,n_elements(var_cal)-1 do begin
        xstr='ncdf_varput,fid,'+var_cal[i]+'_id,'+var_cal[i]
        result=execute(xstr)
      endfor
      ncdf_close,fid
               
      ;****************************
      ;  Calculate cloud layer quantities only on LOW Clouds
      ;****************************

      ;  Flags I don't use
      ;lidar_radar_obs=make_array(numtimes,value=1,/float)

      ;  Ceilometer bases
      ;cloud_base_ceil ;from instrument
      cloud_base_temp=make_array(numtimes,value=missing,/float)
      cloud_base_depol=make_array(numtimes,value=missing,/float)
      ;  Range around ceil base
      cldbase_atten_back=make_array(numtimes,value=missing,/float)
      ;  wacr dbz at ceil base
      cldbase_dbz=make_array(numtimes,value=missing,/float)

      ;  Radar tops
      cloud_top_radar=make_array(numtimes,value=missing,/float)
      cloud_top_radar_dbz=make_array(numtimes,value=missing,/float)
      cloud_top_temp=make_array(numtimes,value=missing,/float)
      radar_top_threshold=make_array(numtimes,value=missing,/float)

      ;  Sonde tops
      cloud_top_sonde=make_array(numtimes,value=missing,/float)
      cloud_top_sonde_temp=make_array(numtimes,value=missing,/float)

      ;  Below cloud base quantities
      ;  average dbz below cloud base
      dbz_avg_subcloud=make_array(numtimes,value=missing,/float)
      ;  average copol below cloud base
      atten_back_max_depol=make_array(numtimes,value=missing,/float)
      ;  average depolarization below cloud base
      depol_below=make_array(numtimes,value=missing,/float)
      ;  average dbz when lidar is above threshold
      dbz_max_depol=make_array(numtimes,value=missing,/float)
      snow_rate=make_array(numtimes,value=missing,/float)
      rain_rate=make_array(numtimes,value=missing,/float)
      
      ;  Plotting only
      cldbase_atten_back_base=make_array(numtimes,/float,value=missing)
      cldbase_atten_back_top=make_array(numtimes,/float,value=missing)
      cldbase_atten_back_ht=make_array(numtimes,/float,value=missing)

      ;  Loop through profiles
      for i=start_j, end_j do begin
      ;for i=0,numtimes-1 do begin
  
        ;  If this is a low cloud defined as lidar cloud base < 4
        if low_cloud_flag[i] eq 1 then begin  
    
        ;  Height index closest to ceil base
        result=min(abs(cloud_base_ceil[i]-height),hidx)
    
        ;  Ceil base temp 
        cloud_base_temp[i]=temp[i,hidx]
    
        ;  Lidar depolarization ratio at ceil base 
        cloud_base_depol[i]=lidar_bscat_crospol[i,hidx]/lidar_bscat_copol[i,hidx]   
    
        ;  Cloud base attenuated backscatter
        ;  Max copol value 5 gates below to 15 gates above the cloud base 
        sidx=hidx-5  & if sidx lt 0 then sidx=0
        eidx=hidx+15  & if eidx gt numheights-1 then eidx=numheights-1
        cldbase_atten_back[i]=max(lidar_bscat_copol[i,sidx:eidx],lidar_max_height)
        cldbase_atten_back_base[i]=height[sidx]
        cldbase_atten_back_top[i]=height[eidx]
        cldbase_atten_back_ht[i]=height[sidx+lidar_max_height]
        
        ;*******************************************
        ;  If the radar saw this base (mask value 2,3,5), look for the radar top
        ;*******************************************
    
        ;  Define radar top threshold in dbz
        radar_top_threshold[i]=-40
    
        ;  Look for radar top
        if cloud_mask_wacr[i,hidx] eq 2 or cloud_mask_wacr[i,hidx] eq 3 or $
           cloud_mask_wacr[i,hidx] eq 5 then begin
          cldbase_dbz[i]=dbz_wacr[i,hidx]  ;cloudbase dbz
          j=hidx
;          while dbz_wacr[i,j] gt radar_top_threshold[i] and j lt numheights-2 do begin  
;            j=j+1
;          endwhile
         while cloud_mask_wacr[i,j] eq 2 or cloud_mask_wacr[i,j] eq 3 or $
            cloud_mask_wacr[i,j] eq 5 do j=j+1
          j=j-1  ;radar cloud top index
          cloud_top_radar[i]=height[j-1]  ;radar top where dbz lt -40
          cloud_top_radar_dbz[i]=dbz_wacr[i,j]
          if cloud_top_radar_dbz[i] lt -100. then stop
          cloud_top_temp[i]=temp[i,j]
          
      max_reflectivity=max(dbz_wacr[i,hidx:j])
      if max_reflectivity gt -10 then begin
        l=where(dbz_wacr[i,hidx:j] eq max_reflectivity) & l=l[0]+hidx
        while dbz_wacr[i,l] gt -25. do begin
          l=l+1
        endwhile
        l=l+1
        cloud_top_radar[i]=height[l-1]  ;radar top where dbz lt -40
        cloud_top_radar_dbz[i]=dbz_wacr[i,l]
        cloud_top_temp[i]=temp[i,l]
      endif          
          
        endif 
    
        ;*******************************************
        ;  Calculate a sounding (inversion) top for each profile
        ;*******************************************
        ;  Clouds usually occur in areas where the dewpoint depression 
        ;  (the difference between the temp and dewpoint) 2-3 degrees C or less.
        dewpoint_depression=temp-dpt

        ;  Sections with rh > 70 are cloud layers
        rh1=rh[i,*] & dewpoint_depression1=dewpoint_depression[i,*]
        result=where(rh1 gt 70 and dewpoint_depression1 lt 3,count)
        if count gt 0 then begin
          ;  Start with the first layer base
          idxs=result[0]
          ;  Loop through and get start and end indexes of layers
          j=0
          for k=0,count-2 do begin
            if result[k+1] gt result[k]+1 then begin
              if j eq 0 then idxe=result[k] else idxe=[idxe,result[k]]
              idxs=[idxs,result[k+1]]
              j=j+1
            endif
          endfor
          if j eq 0 then idxe=result[count-1] else idxe=[idxe,result[count-1]]
        endif
        numlayers=n_elements(idxs)  ;number of cloud layers in soundings
    
        ;  Pick out the sounding layer top closest to the ceil base
        dh=height[idxe]-cloud_base_ceil[i]
        r=where(dh gt 0,c)
        if c gt 0 then begin
          cloud_top_sonde_temp[i]=temp[i,idxe[r[0]]]
          cloud_top_sonde[i]=height[idxe[r[0]]]
        endif
    
        ;*******************************
        ;  below base depol and below base dbz
        ;*******************************
        depol_below[i]=0. ;average lidar depol in 15 gates below 
        dbz_avg_count=0   ;the number of points below cloud base that have qualifying dbz
        atten_back_max_depol[i]=0. ;average copol in 15 gates below
        count_below=0 ;the number of points below cloud base that have qualifying depol
        z_max_depol_sum=0.  ;holds dbz value at max depol value, put in dbz_max_depol
   
        ;  the actual cloud base is several range bins below where the ceilometer is putting it.
        ;  Drop 7 range bins below the lidar cloud base. 7 range bins should usually put you below the actual base.  
        kk=hidx-7  
    
        ;  At least 15 range gates below the cloud
        if kk gt 15 then begin
      
          ;  Found a candidate, initialize value to ~0
          dbz_avg_subcloud[i]=1.e-12
          for kkk=0,14 do begin

            ;  Height index at one of the 15 range gates below ceil base
            result=min(abs(height-height[kk-kkk]),radar_ht_index)
        
            ;  Only sum for radar-lidar cloud
            if cloud_mask_wacr[i,radar_ht_index] eq 2 or cloud_mask_wacr[i,hidx] eq 3 or $
              cloud_mask_wacr[i,hidx] eq 5 then begin    
              dbz_avg_subcloud[i]=dbz_avg_subcloud[i]+(10.^(dbz_wacr[i,radar_ht_index]/10.)) 
              dbz_avg_count=dbz_avg_count+1.
          
              dp=lidar_bscat_crospol[i,kk-kkk]/lidar_bscat_copol[i,kk-kkk]
          
              ;  Signal strength restriction for depolarization
              if cloud_mask_wacr[i,radar_ht_index] eq 2 then begin
                atten_back_max_depol[i]=atten_back_max_depol[i]+lidar_bscat_copol[i,kk-kkk]
                depol_below[i]=depol_below[i]+dp
                z_max_depol_sum=z_max_depol_sum+10.^(dbz_wacr[i,radar_ht_index]/10.)
                count_below=count_below+1.
              endif  ;check signal strength for depol calculation
            endif ;if this is radar-lidar cloud
          endfor ;   for kkk=0,14 do begin
          ;  If there are depol points below
          if count_below gt 0. then begin
            atten_back_max_depol[i]=atten_back_max_depol[i]/count_below
            depol_below[i]=depol_below[i]/count_below
            dbz_max_depol[i]=10.*alog10(z_max_depol_sum/count_below)
            ;if depol_below[i] lt 0.1 then rain_rate[i]=(1./25.)*((10.^(dbz_max_depol[i]/10.))^(1./1.3)) ; mm/hr   Comstock et al. QJ 2004 pg 2891
            if depol_below[i] gt 0.2 then snow_rate[i]=(1./10.)*((10.^(dbz_max_depol[i]/10.))^(1./0.8)) ; mm/hr   Matrosov 2007
          endif
          ;  If there are dbz points below
          if dbz_avg_count gt 0 then begin
            dbz_avg_subcloud[i]=10.*alog10(dbz_avg_subcloud[i]/dbz_avg_count)
            rain_rate[i]=(1./25.)*((10.^(dbz_avg_subcloud[i]/10.))^(1./1.3)) ; mm/hr   Comstock et al. QJ 2004 pg 2891
          endif else begin
            dbz_avg_subcloud[i]=missing  ;Return the value back to missing
          endelse
          ;print,i,dbz_max_depol[i],dbz_avg_subcloud[i],count_below
        endif ;     if kk gt 15 then begin
        ;print, j,cloud_top_temp[j], cloud_base_depol[j], depol_below[j], dbz_avg_subcloud[j]
      endif  ;end of this is a low cloud
      endfor  ;end of loop through shots  

      ;  Variables we want to save from below cloud section
      var_base=['cloud_base_temp','cloud_base_depol','cldbase_atten_back','cldbase_dbz',$
        'cloud_top_radar','cloud_top_radar_dbz','cloud_top_temp','cloud_top_sonde',$
        'cloud_top_sonde_temp','dbz_avg_subcloud','atten_back_max_depol','depol_below',$
        'dbz_max_depol','snow_rate','rain_rate','cloud_base_ceil','liq_mwr','be_lwp',$
        'radar_top_threshold']
        
      name_base=['sonde temperature at ceil base',$ ;cloud_base_temp
        'mpl depolarization at ceil base',$ ;cloud_base_depol
        'mpl max copol value 5 gates below to 15 gates above ceil base',$ ;cldbase_atten_back
        'wacr dbz value at ceil base',$ ;cldbase_dbz
        'height at which the wacr dbz drops below -40 (the radar_top_threshold)',$ ;cloud_top_radar
        'the wacr dbz value at the height at which the wacr dbz drops to below -40',$ ;cloud_top_radar_dbz
        'the sonde temp value at the height at which the wacr dbz drops to below -40',$ ;cloud_top_temp
        'the sonde (inversion) top closest to ceil base',$ ;cloud_top_sonde
        'the temperature at the sonde (inversion) top closest to ceil base',$ ;cloud_top_sonde_temp
        'The average dbz at which cloud mask=2, below ceil base',$ ;dbz_avg_subcloud
        'The average copol at which cloud mask=2 and lidar signal strength is strong, below ceil base',$ ;atten_back_max_depol
        'The average depolarization at which cloud mask=2 and lidar signal strength is strong, below ceil base',$ ;depol_below
        'The average dbz at which cloud mask=2 and lidar signal strength is strong, below ceil base',$ ;dbz_max_depol
        'snow rate from reflectivity relation in Matrosov 2007 table 1 for 94 GHz radar',$ ;snow_rate
        'rain_rate from reflectivity relation in Comstock et al. 2004 QJ pg 2891 ',$ ;rain_rate
        'ceilometer cloud base',$ ;cloud_base_ceil
        'total liquid water along LOS path, marmwrlosM1.b1',$ ;liq_mwr
        'liquid water path best-estimate value, marmwrret1liljclouM1.c2',$ ;be_lwp
        'radar dbz threshold used for determining cloud top'] ;radar_top_threshold
        
      unit_base=['celcius','unitless','NRB cts/(us*km)','dBZ','km','dBZ','celsius','km','celsius',$
        'dBz','NRB cts/(us*km)','unitless','dBZ','mm/hr','mm/hr','km','cm','g/m^2','dBZ']  
        
      ;  Write these cloud base quantities to a file
      fid=ncdf_open(outdir+outname,/write)
      num_did=ncdf_dimid(fid,'numtimes')
      ncdf_control,fid,/redef
      for i=0,n_elements(var_base)-1 do begin
        xstr=var_base[i]+'_id=ncdf_vardef(fid,"'+var_base[i]+'",num_did)'
        result=execute(xstr)
        xstr='ncdf_attput,fid,'+var_base[i]+'_id,"long_name","'+name_base[i]+'"'
        result=execute(xstr)
        xstr='ncdf_attput,fid,'+var_base[i]+'_id,"units","'+unit_base[i]+'"'
        result=execute(xstr)
      endfor
      ncdf_control,fid,/endef
      for i=0,n_elements(var_base)-1 do begin
        xstr='ncdf_varput,fid,'+var_base[i]+'_id,'+var_base[i]
        result=execute(xstr)
      endfor
      ncdf_close,fid
      
      ;****************************************
      ;  Cloud Base Extinction using the Li et al. method
      ;  Estimated cloud base using slope method
      ;  Calculate only for fully attenuated layers
      ;****************************************

      ;  Min copol from ceil base to 100 gates above. 
      min_beta_abv_cldbase=make_array(numtimes,/float,value=missing)  
      ;  Height of the maximum copol in the layer
      copol_max_height=make_array(numtimes,/float,value=missing)
      ;  Simple cloud base at cloud base threshold
      cloud_base_thresh_height=make_array(numtimes,/float,value=missing)

      ;  extinction via slope values
      extinction_slope=make_array(numtimes,/float,value=missing) ;extinction
      extinction_slope_sdv=make_array(numtimes,/float,value=missing)  ;extinction_sdv
      ;  distance above max copol height at which the copol/copol_max falls to exp(-6*eta)
      effective_depth=make_array(numtimes,/float,value=missing)  
      ;  height at which the copol/copol_max falls to exp(-6*eta)
      effective_depth_height=make_array(numtimes,/float,value=missing)
      ;  Save the extinction fit coefficients
      ext_via_slope_fit=make_array(numtimes,2,value=missing,/float)
      ;  This has to be less than 0.1 to get extinction
      chsqr_ext=make_array(numtimes,value=missing,/float)
      lidar_heights_fit=make_array(numtimes,15,value=missing,/float)
      copol_fit=make_array(numtimes,15,value=missing,/float)

      ;  dlnbeta_dz_fit_submaxz values 
      ;  The adjusted cloud base
      cloud_base_estimate=make_array(numtimes,value=missing,/float)
      ;  Save the dlnbeta_dz fit coefficients
      dlnbeta_dz_fit=make_array(numtimes,2,value=missing,/float)
      ;  The slope 
      dlnbeta_dz=make_array(numtimes,/float,value=missing)
      ;  The heights used for the cloud base fit
      dlnbeta_dz_fit_height=make_array(numtimes,2,value=-9999,/float)

      ;  This is the value of copolar atten backscatter that indicates full attenuation 
      ;  of the lidar signal.  Below this value assume fully attenuated.   
      ;noise_abv_thresh=1.e-6  ;capricorn2
      noise_abv_thresh=1e-3


      ;  Loop through the profiles 
      for j=start_j, end_j  do begin
      ;for j=0,numtimes-1 do begin

        ;  hidx=height index closest to ceil base
        result=min(abs(cloud_base_ceil[j]-height),hidx)
  
        ;  Min copol bacscatter from ceil base to 100 gates above
        t=lidar_bscat_copol[j,hidx:hidx+100]
        min_beta_abv_cldbase[j]=min(t[where(t gt 0.)])

        ;  Cloud qualifies for extinction calculation
        if low_cloud_flag[j] eq 1 and max(lidar_bscat_copol[j,*]) gt 0 and $
           eta[j] gt 0 and min_beta_abv_cldbase[j] lt noise_abv_thresh then begin
     
          ;  cidx=height index closest to max copol
          sidx=hidx-20  & if sidx lt 0 then sidx=0
          eidx=hidx+10  & if eidx gt numheights-1 then eidx=numheights-1
          result=max(lidar_bscat_copol[j,sidx:eidx],cidx)
          cidx=cidx+sidx
          copol_max_height[j]=height[cidx]
    
          ;  Simple Threshold base 
          bscat1=reform(lidar_bscat_copol[j,sidx:eidx])
          height1=height[sidx:eidx]
          r=where(bscat1 lt cloud_base_threshold and height1 lt cloud_base_ceil[j],c)
          if c gt 0 then cloud_base_thresh_height[j]=height1[r[-1]]
   
          ;***************************************
          ;  Calculate extinction which is entirely from multiple scattering
          ;  and independent of layer base or top
          ;**************************************  
          ext_fit=[-9999.,-9999.]  ;initialize this value
          num_to_atten=0  ;initialize this value
    
          extinction_via_slope_sb,extinction,$
            eta[j],$  ;-(eta[j]*0.3)
            lidar_bscat_copol[j,cidx:cidx+20],$  ;from max copol to 20gates above
            height[cidx:cidx+20],$
            extinction_sdv,eff_depth,num_to_max,num_to_atten,ext_fit,$
            height_interval,j,chsqr,effective_depth_height1,lidar_heights_fit1,copol_fit1 
          ;  Save fitting values 
          if extinction gt 0. then begin 
          ext_via_slope_fit[j,*]=ext_fit
          chsqr_ext[j]=chsqr
          lidar_heights_fit[j,0:n_elements(lidar_heights_fit1)-1]=lidar_heights_fit1
          copol_fit[j,0:n_elements(copol_fit1)-1]=copol_fit1
          if extinction gt 0. and num_to_atten le 0 then begin
            extinction=-9999.
          endif
          endif
          if extinction gt 0. then begin
            extinction_slope[j]=extinction
            extinction_slope_sdv[j]=extinction_sdv
            effective_depth[j]=eff_depth  ;eff_depth_height-max_copol_height
            effective_depth_height[j]=effective_depth_height1  ;height at exp(-6*eta)
          endif
    
          ;*************************************
          ;  Use fit below to estimate cloud base
          ;*************************************
   
          ;  Uses height in km
          dlnbeta_dz_fit_submaxz_sb,lidar_bscat_copol[j,0:cidx],height[0:cidx],$  
            slope,y_intercept,chsqr,fit_copol,sig_values,i_max,i_down   
          ;  Save fitting values
          dlnbeta_dz_fit[j,1]=slope
          dlnbeta_dz_fit[j,0]=y_intercept
          dlnbeta_dz_fit_height[j,*]=[height[i_down],height[i_max]]
    
          if slope gt 0. then begin
            cloud_base_estimate[j]=(alog(cloud_base_threshold)-y_intercept)/slope ;km
            dlnbeta_dz[j]=slope   
          endif 
    
          ;*************************************
          ;  Profile plot to check the extinction and cloud base fits for extinction and cloud base
          ;*************************************    
          if 1 eq 0 then begin  ;turn of this plot here
      
            ;  Date time string for plot
            caldat,julian_day[j],mm,dd,yy,hh,mi,ss
            datehms_str=string(yy,format='(I4)')+string(mm,format='(I02)')+string(dd,format='(I02)')+$
              '.'+string(hh,format='(I02)')+string(mi,format='(I02)')+string(ss,format='(I02)')
      
            pxdim=1000 & pydim=900
            xl=0.09 & xr=0.96
            yb=0.07 & yt=0.85
            sx=0.06
            sy=0.06
            numplots_x=1
            numplots_y=1
            dx=0.01 & dy=0.01
            position_plots,xl,xr,yb,yt,sx,sy,numplots_x,numplots_y,pos
      
            ;  Height index of cloud base
            closest=min(abs(cloud_base_ceil[j]-height),hidx)
            ;  Height index of cloud top
            if cloud_top_radar[j] eq -8888 then begin
              tidx=hidx+40
            endif else begin
              closest=min(abs(cloud_top_radar[j]-height),tidx)
            endelse
            ;  Height indexes to plot
            shidx=hidx-20
            if shidx lt 0 then shidx=0
            ehidx=tidx+20
            if ehidx ge numheights then ehidx=numheights-1
            x_val=reform(lidar_bscat_copol[j,shidx:ehidx])
            y_val=height[shidx:ehidx]
      
            pnum=0
            p0=plot(x_val,y_val,/buffer,dimensions=[pxdim,pydim],position=pos[pnum,*],$
              xtitle='atten backscatter',ytitle='height',/xstyle,symbol='o',$
              title='atten back profile '+datehms_str,xrange=[1e-3,1e3],sym_size=0.4,$
              yrange=[min(y_val),max(y_val)],/xlog)
            p1=plot([1e-3,1e3],[cloud_base_ceil[j],cloud_base_ceil[j]],/overplot,color='blue')
            p2=plot([1e-3,1e3],[cloud_base_estimate[j],cloud_base_estimate[j]],/overplot,color='magenta')
            p2=plot([1e-3,1e3],[copol_max_height[j],copol_max_height[j]],/overplot,color='orange')
            p2=plot([1e-3,1e3],[cloud_base_thresh_height[j],cloud_base_thresh_height[j]],/overplot,color='saddle brown',linestyle=0)
            p2=plot([1e-3,1e3],[height[i_down],height[i_down]],/overplot,color='green',linestyle=2)
            p2=plot([1e-3,1e3],[height[i_max],height[i_max]],/overplot,color='green',linestyle=2)
            p2=plot([cloud_base_threshold,cloud_base_threshold],[height[shidx],height[ehidx]],/overplot,color='green',linestyle=3)
            p2=plot(exp(copol_fit1),lidar_heights_fit1,/overplot,color='deep sky blue',symbol='o',/sym_filled)
      
            if slope ne -9999. then begin
              p3=plot( [exp((slope*(height[i_down]))+y_intercept),$
                        exp((slope*(height[i_max]))+y_intercept)],$
                       [height[i_down],height[i_max]],color='medium sea green',thick=2,/overplot)
              p3=plot( exp((slope*height)+y_intercept),height,color='medium sea green',thick=2,/overplot,linestyle=2)
            endif 
            t1=text(pos[pnum,0]+2*dx,pos[pnum,3]-8*dy,'ceil base',color='blue')
            t1=text(pos[pnum,0]+2*dx,pos[pnum,3]-10*dy,'max copol height',color='orange')
            t1=text(pos[pnum,0]+2*dx,pos[pnum,3]-12*dy,'estimated base',color='magenta')
      
            if ext_fit[1] ne -9999. then begin  ;y=km
              ii=where(x_val[where(x_val gt 0.)] eq max(x_val[where(x_val gt 0.)]))
              p4=plot([exp((ext_fit[1]*(y_val[ii+2]))+ext_fit[0]),$
                       exp((ext_fit[1]*(y_val[ii+10]))+ext_fit[0])],[y_val[ii+2],y_val[ii+10]],color='deep sky blue',thick=2,/overplot)
            endif
            print,slope,y_intercept,'slope,yintercept for dlndz'
            print,ext_fit[1],ext_fit[0]
            print,exp((ext_fit[1]*(y_val[ii+1]))+ext_fit[0]),exp((ext_fit[1]*(y_val[ii+5]))+ext_fit[0])
            p0.save,outdir+'/profile_plots/test.'+datehms_str+'.'+strcompress(j,/remove_all)+'.png'
          endif  ;  end of do plot
        endif  ;end of its a low cloud
      endfor  ;end of loop through shots

      ;  These are the variables we want to save
      var_fit=['min_beta_abv_cldbase','copol_max_height','cloud_base_thresh_height',$
        'extinction_slope','extinction_slope_sdv','effective_depth','effective_depth_height',$
        'chsqr_ext','cloud_base_estimate','dlnbeta_dz']
      name_fit=['min copol value from ceil base to 100 gates above ceil base',$ ;min_beta_abv_cldbase
        'height of the maximum copol value',$ ;copol_max_height
        'simple cloud base at cloud base threshold',$ ;cloud_base_thresh_height
        'extinction',$ ;extinction_slope
        'extinction standard deviation',$ ;extinction_slope_sdv
        'distance above max copol height at which the copol/copol_max ratio falls to exp(-6*eta)',$ ;effective depth
        'height at which the copol/copol_max ratio falls to exp(-6*eta)',$ ;effective_depth_height
        'value of the unreduced chi-square goodness-of-fit statistic for the extinction fit',$ ;chsqr_fit
        'cloud base from the dlnbeta_dz_fit routine',$ ;cloud_base_estimate
        'slope from the dlnbeta_dz fit routine'] ;dlnbeta_dz
      unit_fit=['Normalized Relative Backscatter (NRB) cts/(us*km)','km','km','/km','/km','km','km','unitless','km','unitless']  

      ;  2d vars 
      ;ext_via_slope_fit  ;extinction fit coefficients
      ;dlnbeta_dz_fit     ;dlnbeta_dz fit coefficients
      ;lidar_heights_fit  ;extinction fit heights
      ;copol_fit          ;extinction fit copol
      ;dlnbeta_dz_fit_height  ;dlnbeta_dz heights
      ;  Constants
      ;noise_abv_thresh       ;value of full attenuation
      ;cloud_base_threshold   ;cloud base backscatter threshold

      ;  Write out the variables to a file
      fid=ncdf_open(outdir+outname,/write)
      num_did=ncdf_dimid(fid,'numtimes')
      ncdf_control,fid,/redef
      fit_did=ncdf_dimdef(fid,'fit',2)
      ext_did=ncdf_dimdef(fid,'ext',15)
      ;  Define variables
      for i=0,n_elements(var_fit)-1 do begin
        xstr=var_fit[i]+'_id=ncdf_vardef(fid,"'+var_fit[i]+'",num_did)'
        result=execute(xstr)
        xstr='ncdf_attput,fid,'+var_fit[i]+'_id,"long_name","'+name_fit[i]+'"'
        result=execute(xstr)
        xstr='ncdf_attput,fid,'+var_fit[i]+'_id,"units","'+unit_fit[i]+'"'
        result=execute(xstr)
      endfor
      ext_via_slope_fit_id=ncdf_vardef(fid,'ext_via_slope_fit',[num_did,fit_did])
      ncdf_attput,fid,ext_via_slope_fit_id,'long_name','extinction fit coefficients'
      ncdf_attput,fid,ext_via_slope_fit_id,'units','unitless'
      dlnbeta_dz_fit_id=ncdf_vardef(fid,'dlnbeta_dz_fit',[num_did,fit_did])
      ncdf_attput,fid,dlnbeta_dz_fit_id,'long_name','dlnbeta_dz fit coefficients'
      ncdf_attput,fid,dlnbeta_dz_fit_id,'units','unitless'
      lidar_heights_fit_id=ncdf_vardef(fid,'lidar_heights_fit',[num_did,ext_did])
      ncdf_attput,fid,lidar_heights_fit_id,'long_name','heights used in the extinction fit'
      ncdf_attput,fid,lidar_heights_fit_id,'units','km'
      copol_fit_id=ncdf_vardef(fid,'copol_fit',[num_did,ext_did])
      ncdf_attput,fid,copol_fit_id,'long_name','copol used in the extinction fit'
      ncdf_attput,fid,copol_fit_id,'units','Normalized Relative Backscatter (NRB) cts/(us*km)'
      dlnbeta_dz_fit_height_id=ncdf_vardef(fid,'dlnbeta_dz_fit_height',[num_did,fit_did])
      ncdf_attput,fid,dlnbeta_dz_fit_height_id,'long_name','heights used in the dlnbeta_dz fit'
      ncdf_attput,fid,dlnbeta_dz_fit_height_id,'units','km'
      noise_abv_thresh_id=ncdf_vardef(fid,'noise_abv_thresh')
      ncdf_attput,fid,noise_abv_thresh_id,'long_name','copol value when mpl is fully attenuated'
      ncdf_attput,fid,noise_abv_thresh_id,'units','Normalized Relative Backscatter (NRB) cts/(us*km)'
      cloud_base_threshold_id=ncdf_vardef(fid,'cloud_base_threshold')
      ncdf_attput,fid,cloud_base_threshold_id,'long_name','copol value at cloud base'
      ncdf_attput,fid,cloud_base_threshold_id,'units','Normalized Relative Backscatter (NRB) cts/(us*km)'
      ncdf_control,fid,/endef
      ;  Write variables
      for i=0,n_elements(var_fit)-1 do begin
        xstr='ncdf_varput,fid,'+var_fit[i]+'_id,'+var_fit[i]
        result=execute(xstr)
      endfor
      ncdf_varput,fid,ext_via_slope_fit_id,ext_via_slope_fit
      ncdf_varput,fid,dlnbeta_dz_fit_id,dlnbeta_dz_fit
      ncdf_varput,fid,lidar_heights_fit_id,lidar_heights_fit
      ncdf_varput,fid,copol_fit_id,copol_fit
      ncdf_varput,fid,dlnbeta_dz_fit_height_id,dlnbeta_dz_fit_height
      ncdf_varput,fid,noise_abv_thresh_id,noise_abv_thresh
      ncdf_varput,fid,cloud_base_threshold_id,cloud_base_threshold
      ncdf_close,fid
      
      ;************************************
      ;  Loop through for FAD 
      ;************************************

      ;  Specific gas constant for dry air  J K-1 kg-1
      Rd=287.04d ;J/kgK
      ;  Latent heat of vaporization J kg-1
      Lv=2.25d6 ;J/kg
      ;  Gravitational acceleration m s-2
      g=9.81d  ; m/s2
      ;  Specific heat of dry air J kg-1 K-1
      cp=1003.5 ; J/kg/K   approximate between 250K and 300 K

      ;  The best base to use for further calculations
      cloud_base=make_array(numtimes,/float,value=missing)
      ;  We need a cloud_base_case flag  1=dlnbeta_dz, 2=simple threshold
      cloud_base_case=make_array(numtimes,value=missing,/int) 
      ;  The best top to use for further calculations
      cloud_top=make_array(numtimes,/float,value=missing)
      ;  We need a cloud_top_case flag  1=radar top, 2=sounding top, 3=add 100m to cloud base
      ;                                  uncert=0.1   uncert=0.3      uncert=0.5
      cloud_top_case=make_array(numtimes,value=missing,/int)
      ;  cloud_top-cloud_base (m)
      cloud_layer_thickness=make_array(numtimes,/float,value=missing)

      ;  moist adiabatic lapse rate
      gamma_m=make_array(numtimes,/float,value=missing)
      ;  condensation rate
      gamma_l=make_array(numtimes,/float,value=missing)
      ;  liq / precip_rate
      water_path_timescale=make_array(numtimes,/float,value=missing)
      ;  adiabatic liquid water path
      adiab_lwp=make_array(numtimes,/float,value=missing)
      ;  FAD from mwr liq/adiabatic liquid water content
      fad_raw=make_array(numtimes,/float,value=missing)
      ;  Modified fad array
      fad=make_array(numtimes,/float,value=missing)
      ;  Fad source  0=no calc  1=mwr/adlwp or raw  2=layer param
      fad_source=make_array(numtimes,/int,value=0)

      ;  Choose the cloud base (km) to use
      ;  If estimated base is less then threshold base, use threshold base
      r=where(cloud_base_estimate lt cloud_base_thresh_height and $
              cloud_base_estimate ne missing and cloud_base_thresh_height ne missing,c)
      if c gt 0 then begin
        cloud_base[r]=cloud_base_thresh_height[r] 
        cloud_base_case[r]=2
      endif
      r=where(cloud_base_estimate ge cloud_base_thresh_height and $
              cloud_base_estimate ne missing and cloud_base_thresh_height ne missing,c)
      if c gt 0 then begin
        cloud_base[r]=cloud_base_estimate[r] 
        cloud_base_case[r]=1
      endif
      
      ;  Choose the cloud top (km) to use
      ;  if radar top is less than ceil base, use sonde top
      ;  If no radar top or sounding top then estimate top to be 100m above cloud base
      cloud_top=cloud_top_radar  ;km
      r=where(cloud_top_radar ne missing,c)
      if c gt 0 then cloud_top_case[r]=1
      r=where( (cloud_top_radar lt cloud_base_ceil or cloud_top_radar gt 8.0) and $
               cloud_top_sonde ne missing and cloud_top_radar ne missing and $
               cloud_base_ceil ne missing,c)
      if c gt 0 then begin
        cloud_top[r]=cloud_top_sonde[r] ;km
        cloud_top_case[r]=2
      endif
      r=where(cloud_top eq missing and cloud_base ne missing,c)
      if c gt 0 then begin
        cloud_top[r]=cloud_base[r]+0.1  ;km
        cloud_top_case[r]=3
      endif
      
      ;  Calculate cloud layer thickness (m)
      r=where(cloud_base ne missing and cloud_top ne missing,c)
      if c gt 0 then cloud_layer_thickness[r]=(cloud_top[r]-cloud_base[r])*1000.0

      ;  Loop through the profiles to calculate fad
      for j=0,numtimes-1 do begin
  
        if low_cloud_flag[j] eq 1 and cloud_layer_thickness[j] gt 0 then begin
  
          ; Cloud base index
          result=min(abs(cloud_base[j]-height),base_index)
  
          ;  Values at cloud base for adiabatic lapse rate
          temp1=temp[j,base_index]+273.15  ;convert C to K
          pres1=pres[j,base_index]*100. ;convert mb to Pa
  
          ;  Moist adiabatic lapse rate 
          ;gamma_m=double(moist_adiabatic_lapse_rate(temp,1.,pressure))/1000.d ;K/m
          gamma_m[j]=double(moist_adiabatic_lapse_rate_jay(temp1,1.,pres1))/1000.d ;K/m
          ;  Condensation rate from Grosvenor et al, 2018 equation 14. kg/m3/m
          cw=(pres1/(rd*temp1))*(cp/lv)*((g/cp)-gamma_m[j])
          gamma_l[j]=cw
          ; calculate the adiabatic lwp
          ;  The integral of x is 0.5*x^2
          adiab_lwp[j]=cw*0.5*((cloud_layer_thickness[j])^2) ; kg per square meter
    
          ;  We aren't using the mwrlos liq value
          if liq_mwr[j] gt 0 and 1 eq 0 then begin
            ;mwr_liq[j]=liq_mwr[j]*1.e4  ;convert cm to g/m2
            if rain_rate[j] gt 0. then water_path_timescale[j]=(liq_mwr[j]*10.)/rain_rate[j]  ; mm/(mm/hr) should return hours
            if snow_rate[j] gt 0. then water_path_timescale[j]=(liq_mwr[j]*10.)/snow_rate[j]  ; mm/(mm/hr) should return hours
            ;  FAD from adiabatic LWP (lwc is a constant fraction (fad) of its adiabatic value)
            fad_raw[j]=(liq_mwr[j]*1.e4)/(adiab_lwp[j]*1.e3)
          endif
    
          ;  Use  the mwrret lwp value instead
          if be_lwp[j] gt 0 and 1 eq 1 then begin
            ;  be_lwp is in g/m2.  multiply by 1e-3 to get mm
            if rain_rate[j] gt 0. then water_path_timescale[j]=(be_lwp[j]*1e-3)/rain_rate[j]  ; mm/(mm/hr) should return hours
            if snow_rate[j] gt 0. then water_path_timescale[j]=(be_lwp[j]*1e-3)/snow_rate[j]  ; mm/(mm/hr) should return hours
            ;  FAD from adiabatic LWP (lwc is a constant fraction (fad) of its adiabatic value)
            fad_raw[j]=(be_lwp[j])/(adiab_lwp[j]*1.e3)
          endif
    
          ;  Only keep raw fad from for certain values, otherwise parameterize it from layer thickness.
          if fad_raw[j] gt 0.4 and fad_raw[j] le 1. then begin
            fad[j]=fad_raw[j]
            fad_source[j]=1
          endif else begin
            if cloud_layer_thickness[j] le 100.0 then begin
              fad[j]=1.0
            endif else if cloud_layer_thickness[j] gt 100.0 and $
                          cloud_layer_thickness[j] le 250.0 then begin
              fad[j]=0.8
            endif else if cloud_layer_thickness[j] gt 250. then begin
              fad[j]=0.52-(0.00046*cloud_layer_thickness[j]) 
              if fad[j] lt 0.5 then fad[j]=0.5
            endif
            fad_source[j]=2
          endelse  
        endif ;end of layer candidate for fad calculation
      endfor  ;end of loop through profiles

      ;  These are the variables from this section saved to the output
      var_fad=['cloud_base','cloud_base_case','cloud_top','cloud_top_case','cloud_layer_thickness',$
        'gamma_m','gamma_l','adiab_lwp','water_path_timescale','fad_raw','fad','fad_source']
      name_fad=['Best cloud base to use.  It is the greater of cloud_base_estimate and cloud_base_thresh_height',$ ;cloud_base
        'Where the base comes from. 1=dlnbeta_dz, 2=simple threshold',$ ;cloud_base_case
        'Best cloud top to use. radar top if it exists, else use sonde top, otherwise use 100m above cloud_base',$ ;cloud_top
        'Where the top comes from. 1=radar top, 2=sounding top, 3=add 100m to cloud base',$ ;cloud_top_case
        'cloud layer thickness',$ ;cloud_layer_thickness
        'moist adiabatic lapse rate',$ ;gamma_m (K/m)
        'condensation rate from Grosvenor et al, 2018 equation 14.',$ ;gamma_l (kg/m3/m)
        'adiabatic liquid water path',$ ;adiab_lwp  (kg/m2)
        'water path timescale',$ ;water_path_timescale
        'fad from (mwr liquid water path) / (adiabatic liquid water path)',$ ;fad_raw
        'fad to use for further calculations, fad_raw for values 0.4 to 1.0. Otherwise parametarize with cloud layer thickness',$ ;fad
        'Where fad comes from. 0=no calc  1=mwr/adlwp or raw  2=layer param'] ;fad_source 
      unit_fad=['km','unitless','km','unitless','meters','K/m','kg/m3/m','kg/m2','hours','unitless','unitless','unitless']   
        
      ;  Write out variables
      fid=ncdf_open(outdir+outname,/write)
      num_did=ncdf_dimid(fid,'numtimes')
      ncdf_control,fid,/redef
      for i=0,n_elements(var_fad)-1 do begin
        xstr=var_fad[i]+'_id=ncdf_vardef(fid,"'+var_fad[i]+'",num_did)'
        result=execute(xstr)
        xstr='ncdf_attput,fid,'+var_fad[i]+'_id,"long_name","'+name_fad[i]+'"'
        result=execute(xstr)
        xstr='ncdf_attput,fid,'+var_fad[i]+'_id,"units","'+unit_fad[i]+'"'
        result=execute(xstr) 
      endfor
      ncdf_control,fid,/endef
      for i=0,n_elements(var_fad)-1 do begin
        xstr='ncdf_varput,fid,'+var_fad[i]+'_id,'+var_fad[i]
        result=execute(xstr)
      endfor
      ncdf_close,fid

      ;*******************************************
      ;*******************************************
      ;  Start nd retrieval
      ;  These variables will be saved to a different file.  
      ;  This is because different numbers of profiles can be 
      ;  used for the running average and we don't want to save the 
      ;  single profile quantities multiple times
      ;*******************************************
      ;*******************************************

      ;  Holds the running means for each profile
      ;  Inputs to the simple nd calculation
      mean_zmax_array=make_array(numtimes,value=missing,/float)
      mean_eta_array=make_array(numtimes,value=missing,/float)
      mean_extinction_array=make_array(numtimes,value=missing,/float)
      mean_lwp_array=make_array(numtimes,value=missing,/float)
      mean_dbz_top_array=make_array(numtimes,value=missing,/float)
      med_cld_dh_array=make_array(numtimes,value=missing,/float)
      opaque_cloud_occurrence=fltarr(numtimes)
      precip_occurrence=fltarr(numtimes)
      mean_fad_array=make_array(numtimes,value=missing,/float)
      ccn_array=make_array(numtimes,value=0,/float)
      ;  Calculation from these inputs
      nd_dlnbeta=make_array(numtimes,value=missing,/float)  ;simple calculation

      ;  Holds the output from the adjust subroutine
      mean_zmax_adj=make_array(numtimes,value=missing,/float)
      mean_fad_adj=make_array(numtimes,value=missing,/float)
      mean_eta_adj=make_array(numtimes,value=missing,/float)
      ;  Calculation from these inputs
      nd_adj=make_array(numtimes,value=missing,/float)  ;first guess

      ;  Output from oe retrieval
      m_test_output=make_array(numtimes,/float,value=missing)
      dbz_top_output=make_array(numtimes,/float,value=missing)
      nd_oe=make_array(numtimes,/float,value=missing)
      nd_oe_unc=make_array(numtimes,/float,value=missing)
      nd_oe_info=make_array(numtimes,/float,value=missing)
      fad_oe=make_array(numtimes,/float,value=missing)
      fad_oe_unc=make_array(numtimes,/float,value=missing)
      fad_oe_info=make_array(numtimes,/float,value=missing)
      re_top_oe=make_array(numtimes,/float,value=missing)
      re_top_oe_unc=make_array(numtimes,/float,value=missing)
      re_top_info_oe=make_array(numtimes,/float,value=missing)
      lwp_oe=make_array(numtimes,/float,value=missing)
      lwp_uncert_oe=make_array(numtimes,/float,value=missing)
      z_max_oe=make_array(numtimes,/float,value=missing)
      z_max_uncer_oe=make_array(numtimes,/float,value=missing)
      dbz_top_uncer_oe=make_array(numtimes,/float,value=missing)
      extinction_oe=make_array(numtimes,/float,value=missing)
      extinction_uncer_oe=make_array(numtimes,/float,value=missing)
      cld_layer_dh_oe=make_array(numtimes,/float,value=missing)
      cld_layer_dh_uncert_oe=make_array(numtimes,/float,value=missing)
      eta_oe=make_array(numtimes,/float,value=missing)
      eta_uncert_oe=make_array(numtimes,/float,value=missing)

      ;number of profiles in the running average - Now set at the top of the program
      ;num_to_avg=40  ;capricorn2
      ;num_to_avg=10.0  ;marcus profiles are already averaged to 30s, so 5 min average
      ;num_to_avg=0.0  ;marcus profiles are already averaged to 30s, so 30s average

;;;;;
;
;      
;      
;      
;      
;      
;      
;      
;      
;      
;      
;      ;  Loop through profiles
;      for j=0,numtimes-1 do begin
        ;for j=720,1080 do begin ; for modis case study at 0720 on 1/5/2018
          ;for j=720,1200 do begin ; for modis case study at 0805 on 2/23/2018
          for j=start_j, end_j do begin
        if j mod 50 eq 0 then print,j,' nd retrieval loop'
            ;if low_cloud_flag[j] eq 1 then stop
        ;  First qualification for Nd retrieval
        if low_cloud_flag[j] eq 1 and $
           max(lidar_bscat_copol[j,*]) gt 0. and $
           eta[j] gt 0 and $
           min_beta_abv_cldbase[j] lt noise_abv_thresh and $
           cloud_base[j] gt 0. then begin  ;and $
           ;julian_day[j] ge julday(2,20,2018,20,15,00) then begin  
    
          ;  Get 'num_to_avg' profiles centered on j
          i0=j-fix(num_to_avg/2.0) & if i0 lt 0 then i0=0
          i1=j+fix(num_to_avg/2.0) & if i1 gt numtimes-1 then i1=numtimes-1
    
          ;  zmax=distance (m) to max copol above cloud base
          h1=copol_max_height[i0:i1]  ;height of max copol (km)
          h2=cloud_base[i0:i1]  ; cloud base (km)
          ;  Calculate zmax
          zmax=h1-h2
          r=where(h1 eq missing or h2 eq missing,c)
          if c gt 0 then zmax[r]=missing 
          ;  Find the mean value
          r=where(zmax gt 0.,c)  ;and max_dbz lt -20.
          if c gt 0 then begin
            mean_zmax=mean(zmax[r])  
            mean_zmax=mean_zmax*1000.0  ;km converted to m
          endif else begin
            mean_zmax=missing
          endelse
          mean_zmax_array[j]=mean_zmax  ;m
    
          ;  Layer effective multiple scattering factor
          e=eta[i0:i1] 
          r=where(e gt 0. and zmax gt 0.,c)  ;and max_dbz lt -20.
          if c gt 0 then begin 
            mean_eta=mean(e[r]) 
          endif else begin
            mean_eta=missing
          endelse
          mean_eta_array[j]=mean_eta
    
          ;  Dbz value below cloud base (of precip)
          max_dbz=dbz_max_depol[i0:i1]  
    
          ;  Extinction
          y=extinction_slope[i0:i1]
          r=where(y gt 0 ,c)  ;and max_dbz lt -20
          if c gt 0 then begin
            mean_extinction=mean(y[r])
          endif else begin
            mean_extinction=missing
          endelse
          mean_extinction_array[j]=mean_extinction

          ;  mean LWP from mwr in g/m2
          m=be_lwp[i0:i1]
          r=where(m gt 0,c)
          if c gt 0 then begin
            mean_lwp=mean(m[r])
          endif else begin
            mean_lwp=missing
          endelse
          mean_lwp_array[j]=mean_lwp
    
          ;  Shape factor for gamma droplet distibution function
          alpha=1.
    
          ;  Dbz at radar top
          y=cloud_top_radar_dbz[i0:i1]
          r=where(y gt missing,c)
          if c gt 0 then begin
            mean_dbz_top=10.*alog10(mean(10.^(y[r]/10.)))
          endif else begin
            mean_dbz_top=missing
          endelse
          mean_dbz_top_array[j]=mean_dbz_top
    
          ;  Cloud layer thickness (m) 
          y=cloud_layer_thickness[i0:i1]
          r=where(y gt 0,c)
          if c gt 0. then begin
            med_cld_dh=median(y[r])
          endif else begin
            med_cld_dh=missing
          endelse
          med_cld_dh_array[j]=med_cld_dh

          result=where(min_beta_abv_cldbase[i0:i1] lt noise_abv_thresh,count)
          opaque_cloud_occurrence[j]=float(count)/float((i1-i0+1))

          result=where(snow_rate[i0:i1] gt 0.004 or rain_rate[i0:i1] gt 0.004,count)
          precip_occurrence[j]=float(count)/float((i1-i0+1))

          ;  FAD, restrict to values 0-2
          y=fad[i0:i1]
          r=where(y gt 0. and y lt 2.,c)
          if c gt 0 then begin
            mean_fad=mean(y[r])
          endif else begin
            mean_fad=missing
          endelse
          mean_fad_array[j]=mean_fad
    
          ;  Index of time in ccn file at least within 1/2 day
          closest=min(abs(julian_day[j]-julday_ccn),ccn_index)
          ;if closest lt 0.5D then ccn_arrray[j]=ccn_0p5_med[ccn_index] ;0.5%ss
          if closest lt 0.5D then begin
            ccn_array[j]=ccn_0p2_med_filter[ccn_index[0]] ;0.2%ss
            ccn_sdv=ccn_0p2_sdv_filter[ccn_index[0]]
            
          endif 
    
          ;  Second qualification for Nd retrieval
          if mean_zmax gt 0. and mean_zmax lt 1000. and $  ;0 < zmax < 1000 meters
             mean_extinction ne missing and $              ;extinction exists                                       
             med_cld_dh gt 0. and med_cld_dh lt 2500. and $;0 < cloud_layer_thickness < 1500 meters                                   
             mean_lwp gt 10. then begin   ;lwp > 10
      
            ;  Initial Nd retrieval calculation
            ;                                                       kg/m3/m -> g/m3/m    
            nd_dlnbeta[j]=nd_dlnbeta_calc(0.,mean_zmax,mean_fad,mean_eta,gamma_l[j]*1000.,alpha)  
            ;print,nd_dlnbeta[j],'first nd calculation'  
      
            ;  If radar did not see the top, assign it dbz=-30  Maybe this should be the radar_top_threshold (-40) instead?
            if mean_dbz_top gt -999. then begin
              ;uncert_dbz=0.25
              uncert_dbz=5.
            endif else if mean_dbz_top le -999. then begin
              uncert_dbz=15.
              mean_dbz_top=-30.0
            endif
            
            ;  Uncertainties
            uncert_ext=0.5 & uncert_zmax=(height_interval*1000.)/mean_zmax
            uncert_lwp=0.2 & uncert_dh=0.2 & uncert_eta=0.3 & uncert_alpha=1.
            
            nd_ret=-1 & uncert_nd=0. & fad_ret=0. & uncert_fad=0.3 & re_top_ret=0. & uncert_re_top=0.
            
            ;  Last section that needs to be checked.  It is turned off right now.
            if 1 eq 0 then begin
              if lwp lt 100. and mean_fad lt 0.8 and precip_occurrence[j] lt 0.1 and $
                opaque_cloud_occurrence[j] gt 0.9 then begin
                print,'going in to adjust mean_fad, med_cld_dh'
                print,precip_occurrence[j],opaque_cloud_occurrence[j]
                mean_fad=1.+((0.8-1.)/(100.))*lwp
                fad_source[j]=3
              endif
            endif

            ;  If there is ccn measurement, change mean_zmax, mean_fad, and mean_eta to get Nd close to ccn measurement
            if ccn_array[j] gt 0 then begin
       
              ;  Storing this adjusted value
              mean_zmax_adj[j]=mean_zmax
              mean_fad_adj[j]=mean_fad
              mean_eta_adj[j]=mean_eta
      
              ;  Calculate Nd from adjusted parameters
              ;                                                       kg/m3/m -> g/m3/m
              nd_adj[j]=nd_dlnbeta_calc(0.,mean_zmax,mean_fad,mean_eta,gamma_l[j]*1000.,alpha)
              ;print,nd_adj[j],'adjusted value'
            endif
            
            ml=mean_lwp
      
            ;  Surface temperature and pressure
            temp1=reform(temp_k[j,*])
            pres1=reform(pres[j,*])
            r=where(temp1 ne missing,c)
            sfc_temp=temp1[r[0]]
            sfc_pres=pres1[r[0]]*100.0
            
            r_rat=0.9

            B=0.0620  ; units of m/kg^1/3
            gamma_eff=gamma_l[j]*(mean_fad) ; units of kg/m3/m
            nd_eff=(r_rat)*ccn_array[j]*0.75*1.e6 ; units of per cubic meter
            dd=med_cld_dh  ; m
            re_wood=(1.e6)*B*(gamma_eff^(1./3.))*(nd_eff^(-1./3.))*(dd^(1./3.)) ; units microns
            
            nd_initial_guess=min([ccn_array[j]*0.75,(nd_dlnbeta[j])])
            if nd_initial_guess lt 20. then nd_initial_guess=ccn_array[j]*0.5

            ;nd_initial_guess=ccn_array[j]
            fad_uncert=0.3 
            if mean_lwp lt 50. then uncert_lwp=0.5
            if mean_lwp lt 20. then uncert_lwp=1.0
            retrieve_nd_fad_re_from_zmax_ext_eta_ztop_nofad_test3,$
              sfc_temp,sfc_pres,$
              cloud_base[j]*1000.0,temp_k[j,*],$
              pres[j,*]*100.0,height*1000.0,$
              mean_dbz_top,uncert_dbz,mean_extinction/1000.,uncert_ext,mean_zmax,uncert_zmax,mean_lwp,uncert_lwp,med_cld_dh,$
              uncert_dh,mean_eta,uncert_eta,alpha,uncert_alpha,mean_fad,fad_uncert, nd_ret,uncert_nd,nd_info,$
              re_top_ret,uncert_re_top,re_top_info,y,fx,info,sx,sy,kx,kb,sb,x_final,$
              num_ret_param,info_vector,S_info,num_ind_obs,m_test,num_iter,fx_vector_uncertainty,$
              nd_initial_guess,nd_initial_guess*0.75, re_wood, 0.25 ;(ccn_array[j]+nd_dlnbeta[j])/2.
              
             
              B=0.0620  ; units of m/kg^1/3 from (3/4Pirho_w)^1/3
            gamma_eff=gamma_l[j]*(mean_fad) ; units of kg/m3/m
            nd_eff=(r_rat)*nd_ret*1.e6 ; units of per cubic meter
            dd=med_cld_dh  ; m
            re_wood=(1.e6)*B*(gamma_eff^(1./3.))*(nd_eff^(-1./3.))*(dd^(1./3.)) ; units microns

              
            if abs(m_test) lt 1. and nd_ret gt 10. and re_top_ret gt 25. and re_wood lt 10. then stop
            ;print,nd_ret,'nd_ret' 
     
            ;  Only saves the arrays if an nd retrieval is done.
            if nd_ret gt 0. then begin
              ;print,'putting these in the oe arrays'
              if abs(m_test) lt 1. then print, re_top_ret, re_wood, nd_ret, mean_lwp
              m_test_output[j]=m_test
              dbz_top_output[j]=mean_dbz_top  ;***;contains the -30 for missing dbz at cloud top
              nd_oe[j]=nd_ret
              nd_oe_unc[j]=uncert_nd
              nd_oe_info[j]=nd_info
              fad_oe[j]=mean_lwp/ml
              fad_oe_unc[j]=uncert_fad
              fad_oe_info[j]=0.
              re_top_oe[j]=min(re_wood,re_top_ret)  ;re_top_ret
              re_top_oe_unc[j]=uncert_re_top
              re_top_info_oe[j]=re_top_info
              lwp_oe[j]=mean_lwp
              lwp_uncert_oe[j]=uncert_lwp
              z_max_oe[j]=mean_zmax
              z_max_uncer_oe[j]=uncert_zmax
              dbz_top_uncer_oe[j]=uncert_dbz  ;***
              extinction_oe[j]=mean_extinction/1000. ;***
              extinction_uncer_oe[j]=uncert_ext
              cld_layer_dh_oe[j]=med_cld_dh
              cld_layer_dh_uncert_oe[j]=uncert_dh
              eta_oe[j]=mean_eta
              eta_uncert_oe[j]=uncert_eta
            endif  ;end of nd_ret > 0
       
          endif  ;Second qualification for retrieval

        endif  ;First qualification for retrieval

      endfor  ;loop through profiles

      ;  These are the variables we want to save from the retrieval
      var_nd=['mean_zmax_array','mean_eta_array','mean_extinction_array',$
        'mean_lwp_array','mean_dbz_top_array','med_cld_dh_array',$
        'opaque_cloud_occurrence','precip_occurrence','mean_fad_array',$
        'ccn_array','nd_dlnbeta','mean_zmax_adj','mean_fad_adj','mean_eta_adj',$
        'nd_adj','m_test_output','dbz_top_output','nd_oe',$
        'nd_oe_unc','nd_oe_info','fad_oe','fad_oe_unc','fad_oe_info',$
        're_top_oe','re_top_oe_unc','re_top_info_oe','lwp_oe',$
        'lwp_uncert_oe','z_max_oe','z_max_uncer_oe','dbz_top_uncer_oe',$
        'extinction_oe','extinction_uncer_oe','cld_layer_dh_oe',$
        'cld_layer_dh_uncert_oe','eta_oe','eta_uncert_oe']
         
      name_nd=['running mean of zmax (distance from cloud base to max copol)',$ ;mean_zmax_array
               'running mean of eta',$ ;mean_eta_array
               'running mean of extinction',$ ;mean_extinction_array
               'running mean of lwp',$ ;mean_lwp_array
               'running mean of cloud top dbz',$ ;mean_dbz_top_array
               'median of cloud layer thickness',$ ;med_cld_dh_array
               '% of profiles in running mean that are attenuated',$ ;opaque_cloud_occurrence
               '% of profiles with precip',$ ;precip_occurrence
               'running mean of fad',$ ;mean_fad_array
               'closest ccn measurement within 12 hours',$ ;ccn_array
               'simple nd calculation from nd_dlnbeta_calc',$ ;nd_dlnbeta
               'mean_zmax from the ajust subroutine to get nd close to ccn',$ ;mean_zmax_adj
               'mean_fad from the adjust subroutine to get nd close to ccn',$ ;mean_fad_adj
               'mean_eta from the adjust subroutine to get close to eta',$ ;mean_eta_adj
               'nd calculation from nd_dlnbeta_calc using the adjusted values',$ ;nd_adj
               'output from nd oe',$ ;m_test_output
               'output from nd oe',$ ;dbz_top_output
               'output from nd oe',$ ;nd_oe
               'output from nd oe',$ ;nd_oe_unc
               'output from nd oe',$ ;nd_oe_info
               'output from nd oe',$ ;fad_oe
               'output from nd oe',$ ;fad_oe_unc
               'output from nd oe',$ ;fad_oe_info
               'output from nd oe',$ ;re_top_oe
               'output from nd oe',$ ;re_top_oe_unc
               'output from nd oe',$ ;re_top_info_oe
               'output from nd oe',$ ;lwp_oe
               'output from nd oe',$ ;lwp_uncert_oe
               'output from nd oe',$ ;z_max_oe
               'output from nd oe',$ ;z_max_uncer_oe
               'output from nd oe',$ ;dbz_top_uncer_oe
               'output from nd oe',$ ;extinction_oe
               'output from nd oe',$ ;extinction_uncer_oe
               'output from nd oe',$ ;output from nd oe',$ ;cld_layer_dh_oe
               'output from nd oe',$ ;cld_layer_dh_uncert_oe
               'output from nd oe',$ ;eta_oe
               'output from nd oe'] ;eta_uncert_oe

      unit_nd=['meters','unitless','/km','g/m2','dBZ','meters','%','%','unitless',$ 
               '#/cm3','#/cm3','meters','unitless','unitless','#/cm3',$ 
               ' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',$ 
               ' ',' ',' ',' ',' ',' ',' '] 

      ;  Put retrieval output in a different file
      outname2=strsplit(outname,'cdf',/regex,/extract)
      avg_time=num_to_avg*30.0
      if avg_time eq 0 then begin
        avg_str='retavg_30sec'
      endif else if avg_time lt 60.0 then begin
        avg_str='retavg_'+string(avg_time,format='(I02)')+'sec'
      endif else begin
        avg_str='retavg_'+string(avg_time/60.0,format='(I02)')+'min'
      endelse  
      outname2=outname2+avg_str+'.cdf'
      print,outname2
      fid=ncdf_create(outdir+outname2,/clobber)
      num_did=ncdf_dimdef(fid,'numtimes',numtimes)
      julian_day_id=ncdf_vardef(fid,'julian_day',num_did,/double)
      ; Write the varible list to file
      for i=0,n_elements(var_nd)-1 do begin
        xstr=var_nd[i]+'_id=ncdf_vardef(fid,"'+var_nd[i]+'",num_did)'
        result=execute(xstr)
        xstr='ncdf_attput,fid,'+var_nd[i]+'_id,"long_name","'+name_nd[i]+'"'
        result=execute(xstr)
        xstr='ncdf_attput,fid,'+var_nd[i]+'_id,"units","'+unit_nd[i]+'"'
        result=execute(xstr)
      endfor
      ncdf_control,fid,/endef
      for i=0,n_elements(var_nd)-1 do begin
        xstr='ncdf_varput,fid,'+var_nd[i]+'_id,'+var_nd[i]
        result=execute(xstr)
      endfor
      ncdf_close,fid
      plot_flag=1
 
    endif  ;30 sec avg file exists for this date

  endif else begin ;instead of running the retrieval, plot an existing file

    ;*********************************************
    ;  Read in existing file to plot and skip running retrieval algorithm
    ;*********************************************
    ;  First read in marcus data
    dumb=file_search('ena_data.'+date_str[ddd]+'.avg.30sec.15m.cdf', count=count_files)
    if count_files gt 0 then begin
      plot_flag=1
    fname='ena_data.'+date_str[ddd]+'.avg.30sec.15m.cdf'
    var=['julian_day','lat','lon','solar_zenith','lidar_bscat_copol',$
      'lidar_bscat_crospol','cloud_base_ceil','detection_status_ceil','sonde_dt',$
      'temp','pres','rh','dp','liq_mwr','cloud_mask_wacr',$
      'dbz_wacr','height']
    fid=ncdf_open(fdir+fname)
    for v=0,n_elements(var)-1 do begin
      xstr='vid=ncdf_varid(fid,"'+var[v]+'")'
      result=execute(xstr)
      xstr='ncdf_varget,fid,vid,'+var[v]
      result=execute(xstr)
    endfor
    ncdf_close,fid
    caldat,julian_day,mm,dd,yy,hh,mi,ss
    print,yy[0],mm[0],dd[0],hh[0],mi[0],ss[0],' radar-lidar data start'
    print,yy[-1],mm[-1],dd[-1],hh[-1],mi[-1],ss[-1],' radar-lidar data start'
    ;  wacr dimensions
    s=size(dbz_wacr,/dimensions)
    numtimes=s[0]
    numheights=s[1]
    ;  Second read in enadata prepped for retreival
    outname='nd_retrieval.ena.'+date_str[ddd]+'.avg.30sec.15m.v2023.cdf'
    vars_cal=['low_cloud_flag','integ_tab','integ_pab','integ_depol','eta','sc','effective_radius','ccc_vector']
    vars_base=['cloud_base_temp','cloud_base_depol','cldbase_atten_back',$
      'cldbase_dbz','cloud_top_radar',$
      'cloud_top_radar_dbz','cloud_top_temp','cloud_top_sonde',$
      'cloud_top_sonde_temp','dbz_avg_subcloud','atten_back_max_depol',$
      'depol_below','dbz_max_depol','snow_rate','rain_rate',$
      'cloud_base_ceil','liq_mwr','be_lwp','radar_top_threshold']
    vars_fit=['min_beta_abv_cldbase','extinction_slope','extinction_slope_sdv',$
      'effective_depth','cloud_base_estimate','dlnbeta_dz','noise_abv_thresh',$
      'dlnbeta_dz_fit','ext_via_slope_fit','chsqr_ext',$
      'cloud_base_threshold','copol_max_height','lidar_heights_fit','copol_fit',$
      'dlnbeta_dz_fit_height','cloud_base_thresh_height']
    vars_fad=['cloud_base','cloud_top','cloud_layer_thickness','gamma_l','gamma_m','water_path_timescale',$
      'adiab_lwp','fad_raw','fad','fad_source']
    var=[vars_cal,vars_base,vars_fit,vars_fad]
    fid=ncdf_open(outdir+outname)
    for v=0,n_elements(var)-1 do begin
      xstr='vid=ncdf_varid(fid,"'+var[v]+'")'
      result=execute(xstr)
      xstr='ncdf_varget,fid,vid,'+var[v]
      result=execute(xstr)
    endfor
    ncdf_close,fid
    ;  Third read in nd retrieval output for the chosen averaging interval
    outname2=strsplit(outname,'cdf',/regex,/extract)
    avg_time=num_to_avg*30.0
    if avg_time eq 0 then begin
      avg_str='retavg_30sec'
    endif else if avg_time lt 60.0 then begin
      avg_str='retavg_'+string(avg_time,format='(I02)')+'sec'
    endif else begin
      avg_str='retavg_'+string(avg_time/60.0,format='(I02)')+'min'
    endelse  
    outname2=outname2+avg_str+'.cdf'
    var_nd=['mean_zmax_array','mean_eta_array','mean_extinction_array',$
        'mean_lwp_array','mean_dbz_top_array','med_cld_dh_array',$
        'opaque_cloud_occurrence','precip_occurrence','mean_fad_array',$
        'ccn_array','nd_dlnbeta','mean_zmax_adj','mean_fad_adj','mean_eta_adj',$
        'nd_adj','m_test_output','dbz_top_output','nd_oe',$
        'nd_oe_unc','nd_oe_info','fad_oe','fad_oe_unc','fad_oe_info',$
        're_top_oe','re_top_oe_unc','re_top_info_oe','lwp_oe',$
        'lwp_uncert_oe','z_max_oe','z_max_uncer_oe','dbz_top_uncer_oe',$
        'extinction_oe','extinction_uncer_oe','cld_layer_dh_oe',$
        'cld_layer_dh_uncert_oe','eta_oe','eta_uncert_oe']
    fid=ncdf_open(outdir+outname2)
    for v=0,n_elements(var_nd)-1 do begin
      xstr='vid=ncdf_varid(fid,"'+var_nd[v]+'")'
      result=execute(xstr)
      xstr='ncdf_varget,fid,vid,'+var_nd[v]
      result=execute(xstr)
    endfor
    ncdf_close,fid
    ;  Rename some vars
    dpt=dp
    lidar_bscat_copol_orig=lidar_bscat_copol
    lidar_bscat_crospol_orig=lidar_bscat_crospol
    ;  This applies the ccc factor to the bscat profiles, but this messes up the plots, so turn off
    if 1 eq 0 then begin
      ;  Find the closest ccc factor
      r=where(ccc_vector ne missing)
      ccc_vector_good=ccc_vector[r]
      julian_day_good=julian_day[r]
      for i=0,numtimes-1 do begin
        ;result=min(abs(julian_day[i]-julian_day_good),cidx)
        ;lidar_bscat_copol[i,*]=lidar_bscat_copol[i,*]*ccc_vector_good[cidx]
        ;lidar_bscat_crospol[i,*]=lidar_bscat_crospol[i,*]*ccc_vector_good[cidx]
        if ccc_vector[i] ne missing then lidar_bscat_copol[i,*]=lidar_bscat_copol[i,*]*ccc_vector[i]
        if ccc_vector[i] ne missing then lidar_bscat_crospol[i,*]=lidar_bscat_crospol[i,*]*ccc_vector[i]
      endfor
    endif
    ;  Add dbz correction to wacr data
    dbz_wacr_orig=dbz_wacr
    r=where(dbz_wacr ne -8888,c)
    if c gt 0 then dbz_wacr[r]=dbz_wacr[r]+5.0
    ;  Read in ena ccn file
    if 1 eq 1 then begin
      ;ccn_dir=path_prefix+'mace-group5/field_programs/marcus/'
      ccn_dir='/Users/u0079358/Documents/marcus/'
      ccn_fname='CCN_MARCUS_1hr.csv'
      read_ccn_marcus_1hr,ccn_dir,ccn_fname,ccn_julian_day,ccn_lat,ccn_lon,ccn_rmsn,$
        ccn_0p1_med,ccn_0p1_mad,ccn_0p1_avg,ccn_0p1_std,ccn_0p1_count,$
        ccn_0p2_med,ccn_0p2_mad,ccn_0p2_avg,ccn_0p2_std,ccn_0p2_count,$
        ccn_0p5_med,ccn_0p5_mad,ccn_0p5_avg,ccn_0p5_std,ccn_0p5_count,$
        ccn_0p8_med,ccn_0p8_mad,ccn_0p8_avg,ccn_0p8_std,ccn_0p8_count,$
        ccn_1p0_med,ccn_1p0_mad,ccn_1p0_avg,ccn_1p0_std,ccn_1p0_count,$
        ccn_0p1_sigma,ccn_0p2_sigma,ccn_0p5_sigma,ccn_0p8_sigma,ccn_1p0
    endif
    endif ; if count gt 0 then begin
  endelse  ;end of read an existing file

  ;***************
  ;  Plotting param that depend on date
  ;***************
  
  if date_str[ddd] eq '20180220' then begin
    max_ht=1.5
    min_val=-1.0
    max_val=1.5
    dmin_radar=-40
    dmax_radar=10.0
    dmin_lwp=0.0
    dmax_lwp=100.0
  endif else if date_str[ddd] eq '20180206' then begin
    max_ht=4.0
    min_val=-2.0
    max_val=2.5
    dmin_radar=-40
    dmax_radar=20.0
    dmin_lwp=0.0
    if count_files gt 0 then dmax_lwp=max(be_lwp)
    if count_files eq 0 then plot_flag=0
  endif else begin
    max_ht=3.0
    min_val=-2.0
    max_val=2.5
    dmin_radar=-40
    dmax_radar=20.0
    dmin_lwp=0.0
    if count_files gt 0 then dmax_lwp=max(be_lwp)
    if count_files eq 0 then plot_flag=0
  endelse
  
  ;******************************************************
  ;  Colortable
  ;******************************************************  
  ;  Top is the last color to scale 256 colors, 0-255
  top_color=252
  ;  Colortable  0-252  
  mytable=colortable(33,ncolors=253)
  ;mytable=colortable(5,ncolors=254)
  ;                           253=white     254=hot pink               ;gray=255
  mytable=[mytable,transpose([255,255,255]),transpose([238,18,137]),transpose([180,180,180])]
  mycbtable=mytable[0:top_color,*]

  ;*********************
  ;  Plot 1 - everything plot
  ;*********************

  ;  Turn on or off plot 1
  plot_flag=0
  if plot_flag eq 1 then begin  ;turn on
  ;if 1 eq 0 then begin  ;turn off
    ;  Set up the plot
    pxdim=1000 & pydim=1000
    xl=0.07 & xr=0.92
    yb=0.05 & yt=0.95
    sx=0.08
    sy=0.017
    numplots_x=2
    numplots_y=7
    position_plots,xl,xr,yb,yt,sx,sy,numplots_x,numplots_y,pos
    ;  Colorbar position
    cbpos=pos
    cbpos[*,0]=pos[*,2]+0.01   ;vertical right side
    cbpos[*,2]=cbpos[*,0]+0.009
    pnum=0
    p0=plot([0,1],[0,1],position=pos[pnum,*],axis_style=4,/nodata,/buffer,dimensions=[pxdim,pydim])
    d=p0.convertcoord([pos[pnum,0],pos[pnum,2]],[pos[pnum,1],pos[pnum,3]],/normal,/to_device)
    isx=(d[0,1,0]-d[0,0,0])
    isy=(d[1,1,0]-d[1,0,0])
    dx=0.01 & dy=0.01
    ;  Date label format
    ;dummy=label_date(date_format=['%D %M %Y!C%H:%I'])
    dummy=label_date(date_format=['%N/%D!C%H:%I'])
    fs1=10
    fs2=12

    ;***
    ;  co-pol reflectivity
    ;***

    pnum=1
    radar_name='MWACR'
    dmin_radar=-40
    dmax_radar=10.0
    r=where(height lt max_ht)  ;max height
    data_var=dbz_wacr[*,r]
    xvar=julian_day
    yvar=height[r]
    image_var=bytscl(data_var,min=dmin_radar,max=dmax_radar,top=top_color)
    r=where(cloud_mask_wacr eq 1,c)
    if c gt 0 then image_var[r]=253
    r=where(data_var le -999,c)
    if c gt 0 then image_var[r]=253
    r=where(data_var eq -8888,c)
    if c gt 0 then image_var[r]=255  ;gray
    i0=image(image_var,/current,position=pos[pnum,*],image_dimensions=[isx,isy],rgb_table=mytable)
    c0=contour(data_var,xvar,yvar,/current,/nodata,position=pos[pnum,*],$
      xtickunits='time',xtickformat='label_date',xrange=[xvar[0],xvar[-1]],yrange=[yvar[0],yvar[-1]],$
      /xstyle,/ystyle,xtickdir=1,ytickdir=1,axis_style=2,$
      ytitle='Height (km)',font_size=fs1,irregular=0)
    p2=plot(xvar,cloud_base_estimate,overplot=c0,color='hot pink',linestyle=6,symbol='o',sym_size=0.1)
    p2=plot(xvar,cloud_top_radar,overplot=c0,color='gray',linestyle=6,symbol='o',sym_size=0.1)
    c0=colorbar(orientation=1,title='dBZe',position=reform(cbpos[pnum,*]),rgb_table=mycbtable,$
      font_size=fs1,tickdir=1,border_on=1,range=[dmin_radar,dmax_radar],textpos=1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]-1.5*dy,radar_name+' Reflectivity',font_size=fs1)
    t1=text(pos[pnum,0]+15.0*dx,pos[pnum,3]-1.5*dy,'base estimate',font_size=fs1,color='hot pink')
    t1=text(pos[pnum,0]+25.0*dx,pos[pnum,3]-1.5*dy,'radar top',font_size=fs1,color='gray')
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'a.',font_size=fs2+1)

    ;***
    ;  co-pol lidar
    ;***

    pnum=3
    data_var=lidar_bscat_copol_orig
    r=where(lidar_bscat_copol_orig gt 0,c)
    if c gt 0 then data_var[r]=alog10(data_var[r])
    r=where(lidar_bscat_copol_orig le 0 and lidar_bscat_copol_orig ne -8888,c)
    if c gt 0 then data_var[r]=-9999
    r=where(height lt max_ht)  ;max height
    data_var=data_var[*,r]
    xvar=julian_day
    yvar=height[r]
    radar_name='MPL'
    min_val=-2.0;-6.
    max_val=1.5;-1.
    image_var=bytscl(data_var,min=min_val,max=max_val,top=top_color)
    ;r=where(data_var eq -9999,c)
    ;if c gt 0 then image_var[r]=253  ;white
    r=where(data_var eq -8888,c)
    if c gt 0 then image_var[r]=255  ;gray
    i0=image(image_var,/current,position=pos[pnum,*],image_dimensions=[isx,isy],rgb_table=mytable)
    c0=contour(data_var,xvar,yvar,/current,/nodata,position=pos[pnum,*],$
      xtickunits='time',xtickformat='(A1)',xrange=[xvar[0],xvar[-1]],yrange=[yvar[0],yvar[-1]],$
      /xstyle,/ystyle,xtickdir=1,ytickdir=1,axis_style=2,$
      ytitle='Height (km)',font_size=fs1,irregular=0)
    p2=plot(xvar,cloud_base_ceil,overplot=c0,color='white',linestyle=6,symbol='o',sym_size=0.1)
    cb0=colorbar(orientation=1,title='log(B)',position=reform(cbpos[pnum,*]),rgb_table=mycbtable,$
      font_size=fs1,tickdir=1,border_on=1,range=[min_val,max_val],textpos=1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]-1.5*dy,radar_name+' Copol Bscat',font_size=fs1)
    t1=text(pos[pnum,0]+14.0*dx,pos[pnum,3]-1.5*dy,'ceil base',font_size=fs1)
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'b.',font_size=fs2+1)

    ;***
    ;  Temperature from sounding
    ;***
    
    pnum=5
    dmin_temp=240
    dmax_temp=280
    temp_k=temp
    r=where(temp_k gt missing,c)
    temp_k[r]=temp_k[r]+273.15
    r=where(height lt max_ht)  ;max height
    xvar=julian_day
    yvar=height[r]
    data_var=temp_k[*,r]
    r=where(data_var gt missing,c)
    dmax_temp=max([dmax_temp,max(data_var[r])])
    dmin_temp=max([dmin_temp,min(data_var[r])])
    image_var=bytscl(data_var,min=dmin_temp,max=dmax_temp,top=top_color)
    ;r=where(data_var le -999,c)
    ;if c gt 0 then image_var[r]=253
    r=where(data_var eq -8888,c)
    if c gt 0 then image_var[r]=255  ;gray
    i0=image(image_var,/current,position=pos[pnum,*],image_dimensions=[isx,isy],rgb_table=mytable)
    c0=contour(data_var,xvar,yvar,/current,/nodata,position=pos[pnum,*],$
      xtickunits='time',xtickformat='(A1)',xrange=[xvar[0],xvar[-1]],yrange=[yvar[0],yvar[-1]],$
      /xstyle,/ystyle,xtickdir=1,ytickdir=1,axis_style=2,$
      ytitle='Height (km)',font_size=fs1,irregular=0)
    p2=plot(xvar,cloud_base_ceil,overplot=c0,color='white',linestyle=6,symbol='o',sym_size=0.1)
    c0=colorbar(orientation=1,title='K',position=reform(cbpos[pnum,*]),rgb_table=mycbtable,$
      font_size=fs1,tickdir=1,border_on=1,range=[dmin_temp,dmax_temp],textpos=1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]-1.5*dy,'Sounding Temperature',font_size=fs1,color='white')
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'a.',font_size=fs2+1)

    ;***
    ;  Cloud base depol and depol below
    ;***

    pnum=2
    p3=plot(julian_day,cloud_base_depol,/current,position=pos[pnum,*],$
      xtickunits='time',xtickformat='(A1)',symbol='.',linestyle=6,sym_size=1.2,$
      ytitle='depol ratio',/xstyle,xrange=[julian_day[0],julian_day[-1]],yrange=[0.,0.6])
    p4=plot(julian_day,depol_below,/overplot,color='red',symbol='.',linestyle=6,sym_size=1.2)
    p4=plot(julian_day,integ_depol,/overplot,color='blue',symbol='.',linestyle=6,sym_size=1.2)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'cloud base depol',font_size=fs1)
    t1=text(pos[pnum,0]+14.0*dx,pos[pnum,3]+0.5*dy,'depol below',font_size=fs1,color='red')
    t1=text(pos[pnum,0]+24.0*dx,pos[pnum,3]+0.5*dy,'layer integ depol',font_size=fs1,color='blue')
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'c.',font_size=fs2+1)

    ;***
    ;  Cloud base temp
    ;***

    pnum=4
    cloud_base_temp_k=cloud_base_temp
    r=where(cloud_base_temp gt missing,c)
    if c gt 0 then cloud_base_temp_k[r]=cloud_base_temp_k[r]+273.15
    cloud_top_temp_k=cloud_top_temp
    r=where(cloud_top_temp gt missing,c)
    if c gt 0 then cloud_top_temp_k[r]=cloud_top_temp_k[r]+273.15
    p5=plot(julian_day,cloud_base_temp_k,/current,position=pos[pnum,*],$
      xtickunits='time',xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='Kelvin',/xstyle,xrange=[julian_day[0],julian_day[-1]],yrange=[280,240])
    p6=plot(julian_day,cloud_top_temp_k,/overplot,color='red',symbol='o',linestyle=6,sym_size=0.1)
    p7=plot([julian_day[0],julian_day[-1]],[273.,273.],/overplot)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'cloud base temp',font_size=fs1)
    t1=text(pos[pnum,0]+14.0*dx,pos[pnum,3]+0.5*dy,'top temp',font_size=fs1,color='red')
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'c.',font_size=fs2+1)

    ;***
    ;  Cloud base dbz
    ;***

    pnum=0
    p8=plot(julian_day,cldbase_dbz,/current,position=pos[pnum,*],$
      xtickunits='time',xtickformat='label_date',symbol='o',linestyle=6,sym_size=0.1,$
      xtitle='Time (UTC)',ytitle='cloud base dbz',$
      xrange=[julian_day[0],julian_day[-1]],yrange=[-40,10])
    p9=plot(julian_day,dbz_max_depol,/overplot,color='red',symbol='o',linestyle=6,sym_size=0.1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'cloud base dbz',font_size=fs1)
    t1=text(pos[pnum,0]+12.0*dx,pos[pnum,3]+0.5*dy,'dbz below base',font_size=fs1,color='red')
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'d.',font_size=fs2+1)

    ;***
    ;  Layer integrated TAB & PAB
    ;***

    pnum=9 
    p10=plot(julian_day,integ_tab,/current,position=pos[pnum,*],$
      xtickunits='time',xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='integrated tab',$
      xrange=[julian_day[0],julian_day[-1]]);,yrange=[0.0,1.0])
    ax=p10.axes
    ax[3].Hide=1  ;hide right y axis
    p11=plot(julian_day,integ_pab,/current,position=pos[pnum,*],$
      symbol='o',linestyle=6,sym_size=0.1,$
      xrange=[julian_day[0],julian_day[-1]],xtickunits='time',axis_style=4,color='red')
    yaxis=axis('Y',location='right',target=p11,title='integrated pab',color='red')  
    ;p11=plot(julian_day,integ_pab,/overplot,color='red',symbol='o',linestyle=6,sym_size=0.1)  
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'Layer Integrated TAB',font_size=fs1)
    t1=text(pos[pnum,0]+17.0*dx,pos[pnum,3]+0.5*dy,'PAB',font_size=fs1,color='red')
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'d.',font_size=fs2+1)

    ;***
    ;  Lidar Ratio and Effective Radius
    ;***
    pnum=7
    p10=plot(julian_day,sc,/current,position=pos[pnum,*],$
      xtickunits='time',xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='Sc',$
      xrange=[julian_day[0],julian_day[-1]],yrange=[15.,23.])
    ax=p10.axes
    ax[3].Hide=1  ;hide right y axis
    p11=plot(julian_day,effective_radius,/current,position=pos[pnum,*],$
      yrange=[-10,40],symbol='o',linestyle=6,sym_size=0.1,$
      xrange=[julian_day[0],julian_day[-1]],xtickunits='time',axis_style=4,color='red')
    yaxis=axis('Y',location='right',target=p11,title='re (microns)',color='red')
    ;oplot, lidar_hrfrac, extinction_slope*10., psym=2, color=50, symsize=0.5
    ;p12=plot(julian_day,extinction_slope,/overplot,color='blue',symbol='o',linestyle=6,sym_size=0.1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'Lidar Ratio (Sc)',font_size=fs1)
    t1=text(pos[pnum,0]+12.0*dx,pos[pnum,3]+0.5*dy,'Effective Radius from Sc calibration',font_size=fs1,color='red')
    ;t1=text(pos[pnum,0]+22.0*dx,pos[pnum,3]-3*dy,'Extinction Slope',font_size=fs1,color='blue')

    ;***
    ;  Extinction slope
    ;***

    pnum=10
    ;if min(extinction_slope) lt 0.1 then extinction_slope[where(extinction_slope le 0.1)]=0.1
    r=where(extinction_slope gt missing,c)
    dmin=min(extinction_slope[r])
    dmax=max(extinction_slope[r])
    if dmax gt 100. then dmax=100.
    p14=plot(julian_day,extinction_slope,/current,position=pos[pnum,*],$;/ylog,$
      ytitle='Extinction Slope (1/km)',$
      xtickunits='time',xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax]);,yrange=[1.,min([300.,max((extinction_slope))])] )
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'layer extinction from slope of signal decay',font_size=fs1)

    ;***
    ;  Corection Factor
    ;***

    r=where(ccc_vector gt 0.,cc_cnt)
    if cc_cnt gt 2 then begin
      pnum=8
      p13=plot(julian_day,ccc_vector,/current,position=pos[pnum,*],$
        xtickunits='time',xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
        ytitle='corr to sc',$
        xrange=[julian_day[0],julian_day[-1]],yrange=[0.,5.*median(ccc_vector[where(ccc_vector gt 0.)])])
      t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'correction need to get Sc of 18.5',font_size=fs1)
    endif  ;more than 2 ccc_vector to plot

    ;***
    ;  Eta, Multiple Scattering Factor
    ;***

    pnum=6
    p13=plot(julian_day,eta_oe,/current,position=pos[pnum,*],$
      xtickunits='time',xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='adjusted eta',$
      xrange=[julian_day[0],julian_day[-1]],yrange=[0.4,1.])
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'Multiple Scattering Factor',font_size=fs1)

    ;***
    ;  Fad and zmax
    ;***

    pnum=11
    r=where(fad gt missing,c)
    dmin=min(fad[r])
    dmax=max(fad[r])
    p11=plot(julian_day,fad,/current,position=pos[pnum,*],$
      xtickunits='time',xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='Fad',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    ax=p11.axes
    ax[3].Hide=1  ;hide right y axis
    r=where(mean_zmax_array gt 0,c)
    dmin=min(mean_zmax_array[r])
    dmax=max(mean_zmax_array[r])
    p12=plot(julian_day,mean_zmax_array,/current,position=pos[pnum,*],$
      symbol='o',linestyle=6,sym_size=0.1,yrange=[dmin,dmax],$
      xrange=[julian_day[0],julian_day[-1]],xtickunits='time',axis_style=4,color='red')
    yaxis=axis('Y',location='right',target=p12,title='Zmax (m)',color='red') 
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'Fad',font_size=fs1)
    t1=text(pos[pnum,0]+15.0*dx,pos[pnum,3]+0.5*dy,'Zmax (meters)',font_size=fs1,color='red')
    ;oplot, lidar_hrfrac,depth_to_max*1000.,psym=-3, color=120, symsize=1.0
    ;oplot, lidar_hrfrac, depth_to_max+depth_atten, psym=1, color=130, symsize=0.5

    ;***
    ;  Cloud Nd
    ;***

    pnum=12
    p12=plot(julian_day,nd_dlnbeta,/current,position=pos[pnum,*],$
      xtickunits='time',xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='Nd (cm-3)',$
      yrange=[1e0,1e4],xrange=[julian_day[0],julian_day[-1]], /ylog)
    ;t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,$
    ;  'Cloud Nd_oe (cm-3) ',font_size=fs1)
    t1=text(pos[pnum,0]+22.0*dx,pos[pnum,3]+0.5*dy,'CCN',font_size=fs1,color='black')  
    t1=text(pos[pnum,0]+25.0*dx,pos[pnum,3]+0.5*dy,'0.5',font_size=fs1,color='green')  
    t1=text(pos[pnum,0]+28.0*dx,pos[pnum,3]+0.5*dy,'0.2',font_size=fs1,color='red')  

    p12a=plot(julian_day,nd_adj,/overplot,color='red',symbol='o',linestyle=6,sym_size=0.1)
    ;12b=plot(julian_day,nd_fad_zmax_uq,/overplot,color='red',symbol='o',linestyle=6,sym_size=0.1)
    if ccn_julian_day ne !NULL then begin 
      p12c=plot(ccn_julian_day,ccn_0p5_med,/overplot,color='green',symbol='x',linestyle=6,sym_size=0.4,sym_thick=3)
      p12c=plot(ccn_julian_day,ccn_0p2_med,/overplot,color='red',symbol='x',linestyle=6,sym_size=0.4,sym_thick=3)
    endif
    imagename=date_str[ddd]+'_lidar_profile_quantities.p1.png'
    p0.save,outdir+imagename

  endif  ;turn off plot 1

  ;*********************
  ;  Plot 2  Diagnose Nd - inputs to nd calculation
  ;*********************

  ;  Turn on or off plot 2
  if plot_flag eq 1 then begin  ;turn on
  ;if 1 eq 0 then begin  ;turn off
    ;  Set up the plot
    pxdim=1000 & pydim=1000
    xl=0.10 & xr=0.92
    yb=0.05 & yt=0.95
    sx=0.08
    sy=0.017
    numplots_x=1
    numplots_y=7
    dx=0.01 & dy=0.01
    position_plots,xl,xr,yb,yt,sx,sy,numplots_x,numplots_y,pos
    ;  Colorbar position
    cbpos=pos
    cbpos[*,0]=pos[*,2]+0.01   ;vertical right side
    cbpos[*,2]=cbpos[*,0]+0.009
    pnum=0
    p0=plot([0,1],[0,1],position=pos[pnum,*],axis_style=4,/nodata,/buffer,dimensions=[pxdim,pydim])
    d=p0.convertcoord([pos[pnum,0],pos[pnum,2]],[pos[pnum,1],pos[pnum,3]],/normal,/to_device)
    isx=(d[0,1,0]-d[0,0,0])
    isy=(d[1,1,0]-d[1,0,0])
    dx=0.01 & dy=0.01
    ;  Date label format
    ;dummy=label_date(date_format=['%D %M %Y!C%H:%I'])
    dummy=label_date(date_format=['%N/%D!C%H:%I'])
    fs1=10
    fs2=12

    ;***
    ;  co-pol reflectivity
    ;***

    pnum=0
    radar_name='MWACR'
    ;dmin_radar=-40.0
    ;dmax_radar=-10.0
    r=where(height lt max_ht)  ;max height
    data_var=dbz_wacr[*,r]
    xvar=julian_day
    yvar=height[r]
    image_var=bytscl(data_var,min=dmin_radar,max=dmax_radar,top=top_color)
    r=where(cloud_mask_wacr eq 1,c)
    if c gt 0 then image_var[r]=253
    r=where(data_var le -999,c)
    if c gt 0 then image_var[r]=253
    r=where(data_var eq -8888,c)
    if c gt 0 then image_var[r]=255  ;gray
    i0=image(image_var,/current,position=pos[pnum,*],image_dimensions=[isx,isy],rgb_table=mytable)
    c0=contour(data_var,xvar,yvar,/current,/nodata,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='label_date',xrange=[xvar[0],xvar[-1]],yrange=[yvar[0],yvar[-1]],$
      /xstyle,/ystyle,xtickdir=1,ytickdir=1,axis_style=2,$
      ytitle='Height (km)',font_size=fs1,irregular=0)
    p2=plot(xvar,cloud_base_estimate,overplot=c0,color='hot pink',linestyle=6,symbol='o',sym_size=0.1)
    p2=plot(xvar,cloud_top_radar,overplot=c0,color='gray',linestyle=6,symbol='o',sym_size=0.1)
    c0=colorbar(orientation=1,title='dBZe',position=reform(cbpos[pnum,*]),rgb_table=mycbtable,$
      font_size=fs1,tickdir=1,border_on=1,range=[dmin_radar,dmax_radar],textpos=1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]-1.5*dy,radar_name+' Reflectivity',font_size=fs1)
    t1=text(pos[pnum,0]+15.0*dx,pos[pnum,3]-1.5*dy,'base estimate',font_size=fs1,color='hot pink')
    t1=text(pos[pnum,0]+25.0*dx,pos[pnum,3]-1.5*dy,'radar top',font_size=fs1,color='gray')
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'a.',font_size=fs2+1)

    ;***
    ;  co-pol lidar
    ;***

    pnum=1
    data_var=lidar_bscat_copol_orig
    r=where(lidar_bscat_copol_orig gt 0,c)
    if c gt 0 then data_var[r]=alog10(data_var[r])
    r=where(lidar_bscat_copol_orig le 0 and lidar_bscat_copol_orig ne -8888,c)
    if c gt 0 then data_var[r]=-9999
    r=where(height lt max_ht)  ;max height
    data_var=data_var[*,r]
    xvar=julian_day
    yvar=height[r]
    radar_name='MPL'
    ;min_val=-1.0
    ;max_val=1.0
    image_var=bytscl(data_var,min=min_val,max=max_val,top=top_color)
    ;r=where(data_var eq -9999,c)
    ;if c gt 0 then image_var[r]=253  ;white
    r=where(data_var eq -8888,c)
    if c gt 0 then image_var[r]=255  ;gray
    i0=image(image_var,/current,position=pos[pnum,*],image_dimensions=[isx,isy],rgb_table=mytable)
    c0=contour(data_var,xvar,yvar,/current,/nodata,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',xrange=[xvar[0],xvar[-1]],yrange=[yvar[0],yvar[-1]],$
      /xstyle,/ystyle,xtickdir=1,ytickdir=1,axis_style=2,$
      ytitle='Height (km)',font_size=fs1,irregular=0)
    p2=plot(xvar,cloud_base_ceil,overplot=c0,color='white',linestyle=6,symbol='o',sym_size=0.1)
    cb0=colorbar(orientation=1,title='log(B)',position=reform(cbpos[pnum,*]),rgb_table=mycbtable,$
      font_size=fs1,tickdir=1,border_on=1,range=[min_val,max_val],textpos=1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]-1.5*dy,radar_name+' Copol Bscat',font_size=fs1)
    t1=text(pos[pnum,0]+14.0*dx,pos[pnum,3]-1.5*dy,'ceil base',font_size=fs1)
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'b.',font_size=fs2+1)

    ;***
    ;  Cloud Nd
    ;***

    pnum=2
    r=where(nd_dlnbeta ne missing and finite(nd_dlnbeta),c)
    dmin=min(nd_dlnbeta[r])
    dmax=max(nd_dlnbeta[r])
    p12=plot(julian_day,nd_dlnbeta,/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ;yrange=[dmin,dmax],/ylog,$
      yrange=[1e0,1e4],/ylog,$
      ytitle='Nd (cm-3)',xrange=[julian_day[0],julian_day[-1]])
    p12a=plot(julian_day,nd_oe,/overplot,color='blue',symbol='o',linestyle=6,sym_size=0.1)
;    p12a=plot(ccn_julian_day,ccn_0p5_med,/overplot,color='red',symbol='x',linestyle=6,sym_size=0.4,sym_thick=2) 
    if max(julday_ccn) gt 0. then begin
    p12a=plot(julday_ccn,ccn_0p2_med_filter,/overplot,color='green',symbol='x',linestyle=6,sym_size=0.4,sym_thick=2)  
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'Cloud Nd (cm-3)',font_size=fs1)
    t1=text(pos[pnum,0]+15.0*dx,pos[pnum,3]+0.5*dy,'CCN 0.5 median',font_size=fs1,color='red')
    t1=text(pos[pnum,0]+30.0*dx,pos[pnum,3]+0.5*dy,'CCN 0.2 median',font_size=fs1,color='green')
    t1=text(pos[pnum,0]+45.0*dx,pos[pnum,3]+0.5*dy,'Cloud Nd OE (cm-3)',color='blue',font_size=fs1)
    endif
    ;***
    ;  Fad 
    ;***

    pnum=3
    r=where(mean_fad_array gt missing,c)
    dmin=0.0;min(fad_raw[r])
    dmax=1.5;max(fad_raw[r])
    p11=plot(julian_day,mean_fad_array,/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='Fad',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    ;p11a=plot(julian_day,fad,/overplot,color='green',symbol='o',linestyle=6,sym_size=0.3)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'fad ',font_size=fs1)
    ;t1=text(pos[pnum,0]+10.0*dx,pos[pnum,3]+0.5*dy,'mwr_liq/adiab_lwp ',font_size=fs1)

    ;***
    ;  zmax
    ;***

    pnum=4
    r=where(mean_zmax_array gt missing and mean_zmax_array lt 2000.0,c)
    dmin=min(mean_zmax_array[r])
    dmax=max(mean_zmax_array[r])
    p11=plot(julian_day,mean_zmax_array,/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='Zmax',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'Zmax (meters)',font_size=fs1)
    ;t1=text(pos[pnum,0]+10.0*dx,pos[pnum,3]+0.5*dy,'mwr_liq/adiab_lwp ',font_size=fs1)

    ;***
    ;  ETA
    ;***

    pnum=5
    dmin=min(eta_oe[r]) & if dmin lt 0. then dmin=0.
    dmax=max(eta_oe[r])
    p11=plot(julian_day,mean_eta_array,/current,position=pos[pnum,*],color='black',$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='Adjusted Eta',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'Eta',font_size=fs1,color='black')

    ;***
    ;  gamma_l
    ;***

    pnum=6
    xvar=gamma_l
    r=where(gamma_l gt 0, count_gamma_l)
    if count_gamma_l gt 0 then begin
    xvar[r]=xvar[r]*1000.  ;convert units
    dmin=min(xvar[r])
    dmax=max(xvar[r])
    p06=plot(julian_day[r],xvar[r],/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='gamma_l',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'condensation rate for adiabatic ascent (g/m3/m)',font_size=fs1)  
    endif
    
    xvar=gamma_l
    r=where(gamma_l gt 0, count_gamma_l)
    if count_gamma_l gt 0 then begin
      xvar[r]=xvar[r]*1000.  ;convert units
      dmin=min(xvar[r])
      dmax=max(xvar[r])
      p06=plot(julian_day[r],xvar[r],/current,position=pos[pnum,*],$
        xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
        ytitle='gamma_l',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
      t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'condensation rate for adiabatic ascent (g/m3/m)',font_size=fs1)
    endif
  
    t1=text(0.1,0.98,program+' '+date_str[ddd]+' '+avg_str,font_size=14)
    ;t1=text(0.3,0.97,'solar zenith angle='+string(mean(solar_zenith),format='(F7.2)'),font_size=14 )
    ;t1=text(0.7,0.97,'lat='+string(lat[0],format='(F6.2)'),font_size=14)
    ;t1=text(0.84,0.97,'lon='+string(lon[0],format='(F6.2)'),font_size=14)  
 
    imagename=date_str[ddd]+'_lidar_profile_quantities.'+avg_str+'.p2.png'
    p0.save,outdir+imagename

  endif  ;turn off plot2
  
  ;*********************
  ;  Plot 2.5  Diagnose Nd - inputs to nd calculation
  ;*********************
plot_flag=1
  ;  Turn on or off plot 2
  if plot_flag eq 1 and n_elements(cloud_base_estimate) gt 0 then begin  ;turn on
    ;if 1 eq 0 then begin  ;turn off
    ;  Set up the plot
    pxdim=1000 & pydim=1000
    xl=0.10 & xr=0.92
    yb=0.05 & yt=0.95
    sx=0.08
    sy=0.017
    numplots_x=1
    numplots_y=7
    dx=0.01 & dy=0.01
    position_plots,xl,xr,yb,yt,sx,sy,numplots_x,numplots_y,pos
    ;  Colorbar position
    cbpos=pos
    cbpos[*,0]=pos[*,2]+0.01   ;vertical right side
    cbpos[*,2]=cbpos[*,0]+0.009
    pnum=0
    p0=plot([0,1],[0,1],position=pos[pnum,*],axis_style=4,/nodata,/buffer,dimensions=[pxdim,pydim])
    d=p0.convertcoord([pos[pnum,0],pos[pnum,2]],[pos[pnum,1],pos[pnum,3]],/normal,/to_device)
    isx=(d[0,1,0]-d[0,0,0])
    isy=(d[1,1,0]-d[1,0,0])
    dx=0.01 & dy=0.01
    ;  Date label format
    ;dummy=label_date(date_format=['%D %M %Y!C%H:%I'])
    dummy=label_date(date_format=['%N/%D!C%H:%I'])
    fs1=10
    fs2=12

    ;***
    ;  co-pol reflectivity
    ;***

    pnum=0
    radar_name='MWACR'
    ;dmin_radar=-40.0
    ;dmax_radar=-10.0
    r=where(height lt max_ht)  ;max height
    data_var=dbz_wacr[*,r]
    xvar=julian_day
    yvar=height[r]
    image_var=bytscl(data_var,min=dmin_radar,max=dmax_radar,top=top_color)
    r=where(cloud_mask_wacr eq 1,c)
    if c gt 0 then image_var[r]=253
    r=where(data_var le -999,c)
    if c gt 0 then image_var[r]=253
    r=where(data_var eq -8888,c)
    if c gt 0 then image_var[r]=255  ;gray
    i0=image(image_var,/current,position=pos[pnum,*],image_dimensions=[isx,isy],rgb_table=mytable)
    c0=contour(data_var,xvar,yvar,/current,/nodata,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='label_date',xrange=[xvar[0],xvar[-1]],yrange=[yvar[0],yvar[-1]],$
      /xstyle,/ystyle,xtickdir=1,ytickdir=1,axis_style=2,$
      ytitle='Height (km)',font_size=fs1,irregular=0)
    p2=plot(xvar,cloud_base_estimate,overplot=c0,color='hot pink',linestyle=6,symbol='o',sym_size=0.1)
    p2=plot(xvar,cloud_top_radar,overplot=c0,color='gray',linestyle=6,symbol='o',sym_size=0.1)
    c0=colorbar(orientation=1,title='dBZe',position=reform(cbpos[pnum,*]),rgb_table=mycbtable,$
      font_size=fs1,tickdir=1,border_on=1,range=[dmin_radar,dmax_radar],textpos=1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]-1.5*dy,radar_name+' Reflectivity',font_size=fs1)
    t1=text(pos[pnum,0]+15.0*dx,pos[pnum,3]-1.5*dy,'base estimate',font_size=fs1,color='hot pink')
    t1=text(pos[pnum,0]+25.0*dx,pos[pnum,3]-1.5*dy,'radar top',font_size=fs1,color='gray')
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'a.',font_size=fs2+1)

    ;***
    ;  co-pol lidar
    ;***

    pnum=1
    data_var=lidar_bscat_copol_orig
    r=where(lidar_bscat_copol_orig gt 0,c)
    if c gt 0 then data_var[r]=alog10(data_var[r])
    r=where(lidar_bscat_copol_orig le 0 and lidar_bscat_copol_orig ne -8888,c)
    if c gt 0 then data_var[r]=-9999
    r=where(height lt max_ht)  ;max height
    data_var=data_var[*,r]
    xvar=julian_day
    yvar=height[r]
    radar_name='MPL'
    ;min_val=-1.0
    ;max_val=1.0
    image_var=bytscl(data_var,min=min_val,max=max_val,top=top_color)
    ;r=where(data_var eq -9999,c)
    ;if c gt 0 then image_var[r]=253  ;white
    r=where(data_var eq -8888,c)
    if c gt 0 then image_var[r]=255  ;gray
    i0=image(image_var,/current,position=pos[pnum,*],image_dimensions=[isx,isy],rgb_table=mytable)
    c0=contour(data_var,xvar,yvar,/current,/nodata,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',xrange=[xvar[0],xvar[-1]],yrange=[yvar[0],yvar[-1]],$
      /xstyle,/ystyle,xtickdir=1,ytickdir=1,axis_style=2,$
      ytitle='Height (km)',font_size=fs1,irregular=0)
    p2=plot(xvar,cloud_base_ceil,overplot=c0,color='white',linestyle=6,symbol='o',sym_size=0.1)
    cb0=colorbar(orientation=1,title='log(B)',position=reform(cbpos[pnum,*]),rgb_table=mycbtable,$
      font_size=fs1,tickdir=1,border_on=1,range=[min_val,max_val],textpos=1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]-1.5*dy,radar_name+' Copol Bscat',font_size=fs1)
    t1=text(pos[pnum,0]+14.0*dx,pos[pnum,3]-1.5*dy,'ceil base',font_size=fs1)
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'b.',font_size=fs2+1)

    ;***
    ;  Cloud Nd
    ;***

    pnum=2
    r=where(nd_dlnbeta ne missing and finite(nd_dlnbeta),c)
    dmin=min(nd_dlnbeta[r])
    dmax=max(nd_dlnbeta[r])
    p12=plot(julian_day,nd_dlnbeta,/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ;yrange=[dmin,dmax],/ylog,$
      yrange=[1e0,1e4],/ylog,$
      ytitle='Nd (cm-3)',xrange=[julian_day[0],julian_day[-1]])
    p12a=plot(julian_day,nd_oe,/overplot,color='blue',symbol='o',linestyle=6,sym_size=0.1)
    ;    p12a=plot(ccn_julian_day,ccn_0p5_med,/overplot,color='red',symbol='x',linestyle=6,sym_size=0.4,sym_thick=2)
    if max(julday_ccn) gt 0. then begin
      p12a=plot(julday_ccn,ccn_0p2_med_filter,/overplot,color='green',symbol='x',linestyle=6,sym_size=0.4,sym_thick=2)
      t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'Cloud Nd (cm-3)',font_size=fs1)
      t1=text(pos[pnum,0]+15.0*dx,pos[pnum,3]+0.5*dy,'CCN 0.5 median',font_size=fs1,color='red')
      t1=text(pos[pnum,0]+30.0*dx,pos[pnum,3]+0.5*dy,'CCN 0.2 median',font_size=fs1,color='green')
      t1=text(pos[pnum,0]+45.0*dx,pos[pnum,3]+0.5*dy,'Cloud Nd OE (cm-3)',color='blue',font_size=fs1)
    endif
    ;***
    ;  Fad
    ;***

    pnum=3
    r=where(cloud_top_radar_dbz gt missing,c)
    dmin=-30.;min(fad_raw[r])
    dmax=20.;max(fad_raw[r])
    p11=plot(julian_day,cloud_top_radar_dbz,/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='cloud top dBZ',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    ;p11a=plot(julian_day,fad,/overplot,color='green',symbol='o',linestyle=6,sym_size=0.3)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'cloud top dBZ ',font_size=fs1)
    ;t1=text(pos[pnum,0]+10.0*dx,pos[pnum,3]+0.5*dy,'mwr_liq/adiab_lwp ',font_size=fs1)

    ;***
    ;  zmax
    ;***

    pnum=4
    r=where(dbz_avg_subcloud gt -30.,c)
    dmin=min(dbz_avg_subcloud[r])
    dmax=max(dbz_avg_subcloud[r])
    p11=plot(julian_day,dbz_avg_subcloud,/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='dbz_avg_subcloud',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'dbz_avg_subcloud',font_size=fs1)
    ;t1=text(pos[pnum,0]+10.0*dx,pos[pnum,3]+0.5*dy,'mwr_liq/adiab_lwp ',font_size=fs1)

    ;***
    ;  ETA
    ;***

    pnum=5
    dmin=min(rain_rate[r]*24.) & if dmin lt 0. then dmin=0.
    dmax=max(rain_rate[r]*24.)
    p11=plot(julian_day,rain_rate*24.,/current,position=pos[pnum,*],color='black',$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='rain rate mm/day',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'rain rate (mm/day)',font_size=fs1,color='black')

    ;***
    ;  gamma_l
    ;***

    pnum=6
    xvar=re_top_oe
    r=where(re_top_oe gt 0., count_gamma_l)
    if count_gamma_l gt 0 then begin
      ;xvar[r]=xvar[r]*1000.  ;convert units
      dmin=min(xvar[r])
      dmax=40.  ;max(xvar[r])
      p06=plot(julian_day[r],xvar[r],/current,position=pos[pnum,*],$
        xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
        ytitle='re top',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
      t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'re top (microns)',font_size=fs1)
    endif

;    xvar=gamma_l
;    r=where(gamma_l gt 0, count_gamma_l)
;    if count_gamma_l gt 0 then begin
;      xvar[r]=xvar[r]*1000.  ;convert units
;      dmin=min(xvar[r])
;      dmax=max(xvar[r])
;      p06=plot(julian_day[r],xvar[r],/current,position=pos[pnum,*],$
;        xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
;        ytitle='gamma_l',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
;      t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]+0.5*dy,'condensation rate for adiabatic ascent (g/m3/m)',font_size=fs1)
;    endif

    t1=text(0.1,0.98,program+' '+date_str[ddd]+' '+avg_str,font_size=14)
    ;t1=text(0.3,0.97,'solar zenith angle='+string(mean(solar_zenith),format='(F7.2)'),font_size=14 )
    ;t1=text(0.7,0.97,'lat='+string(lat[0],format='(F6.2)'),font_size=14)
    ;t1=text(0.84,0.97,'lon='+string(lon[0],format='(F6.2)'),font_size=14)

    imagename=date_str[ddd]+'_lidar_profile_quantities.'+avg_str+'.p25.png'
    p0.save,outdir+imagename

  endif  ;turn off plot2

  ;*********************
  ;*********************
  ;  Plot 3  FAD Calculation
  ;*********************
  ;*********************
plot_flag=0
  ;  Turn on or off plot 3
  if plot_flag eq 1 then begin

    ;  Set up the plot
    pxdim=1000 & pydim=1000
    xl=0.08 & xr=0.85
    yb=0.05 & yt=0.95
    sx=0.08
    sy=0.017
    numplots_x=1
    numplots_y=7
    dx=0.01 & dy=0.01
    position_plots,xl,xr,yb,yt,sx,sy,numplots_x,numplots_y,pos
    ;  Colorbar position
    cbpos=pos
    cbpos[*,0]=pos[*,2]+0.01   ;vertical right side
    cbpos[*,2]=cbpos[*,0]+0.009
    pnum=0
    p0=plot([0,1],[0,1],position=pos[pnum,*],axis_style=4,/nodata,/buffer,dimensions=[pxdim,pydim])
    d=p0.convertcoord([pos[pnum,0],pos[pnum,2]],[pos[pnum,1],pos[pnum,3]],/normal,/to_device)
    isx=(d[0,1,0]-d[0,0,0])
    isy=(d[1,1,0]-d[1,0,0])
    dx=0.01 & dy=0.01
    ;  Date label format
    ;dummy=label_date(date_format=['%D %M %Y!C%H:%I'])
    dummy=label_date(date_format=['%N/%D!C%H:%I'])
    fs1=10
    fs2=12


    ;***
    ;  co-pol reflectivity
    ;***

    pnum=0
    radar_name='MWACR'
    ;dmin_radar=-40
    ;dmax_radar=5.0
    r=where(height lt max_ht)  ;max height
    data_var=dbz_wacr[*,r]
    xvar=julian_day
    yvar=height[r]
    image_var=bytscl(data_var,min=dmin_radar,max=dmax_radar,top=top_color)
    r=where(cloud_mask_wacr eq 1,c)
    if c gt 0 then image_var[r]=253
    r=where(data_var le -999,c)
    if c gt 0 then image_var[r]=253
    r=where(data_var eq -8888,c)
    if c gt 0 then image_var[r]=255  ;gray
    i0=image(image_var,/current,position=pos[pnum,*],image_dimensions=[isx,isy],rgb_table=mytable)
    c0=contour(data_var,xvar,yvar,/current,/nodata,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='label_date',xrange=[xvar[0],xvar[-1]],yrange=[yvar[0],yvar[-1]],$
      /xstyle,/ystyle,xtickdir=1,ytickdir=1,axis_style=2,$
      ytitle='Height (km)',font_size=fs1,irregular=0)
    p1=plot(xvar,cloud_base_estimate,overplot=c0,color='hot pink',linestyle=6,symbol='o',sym_size=0.1)
    p2=plot(xvar,cloud_top_radar,overplot=c0,color='gray',linestyle=6,symbol='o',sym_size=0.1)
    c0=colorbar(orientation=1,title='dBZe',position=reform(cbpos[pnum,*]),rgb_table=mycbtable,$
      font_size=fs1,tickdir=1,border_on=1,range=[dmin_radar,dmax_radar],textpos=1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]-1.5*dy,radar_name+' Reflectivity',font_size=fs1)
    t1=text(pos[pnum,0]+15.0*dx,pos[pnum,3]-1.5*dy,'base estimate',font_size=fs1,color='hot pink')
    t1=text(pos[pnum,0]+25.0*dx,pos[pnum,3]-1.5*dy,'radar top',font_size=fs1,color='gray')
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'a.',font_size=fs2+1)

    ;***
    ;  co-pol lidar
    ;***

    pnum=1
    data_var=lidar_bscat_copol_orig
    r=where(lidar_bscat_copol_orig gt 0,c)
    if c gt 0 then data_var[r]=alog10(data_var[r])
    r=where(lidar_bscat_copol_orig le 0 and lidar_bscat_copol_orig ne -8888,c)
    if c gt 0 then data_var[r]=-9999
    r=where(height lt max_ht)  ;max height
    data_var=data_var[*,r]
    xvar=julian_day
    yvar=height[r]
    radar_name='MPL'
    ;min_val=-1.0
    ;max_val=1.0
    image_var=bytscl(data_var,min=min_val,max=max_val,top=top_color)
    ;r=where(data_var eq -9999,c)
    ;if c gt 0 then image_var[r]=253  ;white
    r=where(data_var eq -8888,c)
    if c gt 0 then image_var[r]=255  ;gray
    i0=image(image_var,/current,position=pos[pnum,*],image_dimensions=[isx,isy],rgb_table=mytable)
    c0=contour(data_var,xvar,yvar,/current,/nodata,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',xrange=[xvar[0],xvar[-1]],yrange=[yvar[0],yvar[-1]],$
      /xstyle,/ystyle,xtickdir=1,ytickdir=1,axis_style=2,$
      ytitle='Height (km)',font_size=fs1,irregular=0)
    p2=plot(xvar,cloud_base_ceil,overplot=c0,color='white',linestyle=6,symbol='o',sym_size=0.1)
    cb0=colorbar(orientation=1,title='log(B)',position=reform(cbpos[pnum,*]),rgb_table=mycbtable,$
      font_size=fs1,tickdir=1,border_on=1,range=[min_val,max_val],textpos=1)
    t1=text(pos[pnum,0]+1.0*dx,pos[pnum,3]-1.5*dy,radar_name+' Copol Bscat',font_size=fs1)
    t1=text(pos[pnum,0]+14.0*dx,pos[pnum,3]-1.5*dy,'ceil base',font_size=fs1)
    ;t2=text(pos[pnum,0]-7.0*dx,pos[pnum,3]-1.0*dy,'b.',font_size=fs2+1)

    ;***
    ;  LWP
    ;***

    pnum=2
    var=be_lwp
    r=where(fad_raw eq missing,c)
    if c gt 0 then var[r]=missing
    ;dmin_lwp=0.0
    ;dmax_lwp=100.0
    p3=plot(julian_day,var,/current,position=pos[pnum,*],$
      symbol='o',linestyle=6,sym_size=0.2,yrange=[dmin_lwp,dmax_lwp],$
      xrange=[julian_day[0],julian_day[-1]],xtickunits='hours',$
      xtickinterval=2,color='red',ytitle='LWP g/m2')
    var=adiab_lwp
    r=where(fad_raw eq missing,c)
    if c gt 0 then var[r]=missing
    r=where(var ne missing,c)
    if c gt 0 then var[r]=var[r]*1e3  ;kg/m2 to g/m2
    ;dmin=min(var[r])
    ;dmax=max(var[r])
    p4=plot(julian_day,var,/overplot,symbol='o',linestyle=6,sym_size=0.1,$
      color='blue')
    t1=text(pos[pnum,0]+2.0*dx,pos[pnum,3]-2.0*dy,'BE RET LWP',font_size=fs1,color='red')
    t1=text(pos[pnum,0]+12.0*dx,pos[pnum,3]-2.0*dy,'Adiabatic LWP',color='blue',font_size=fs1) 
 
    ;***
    ;  Fad raw ;fad_raw[j]=(liq_mwr[j]*1.e4)/(adiab_lwp[j]*1.e3)
    ;***

    pnum=3
    var=fad_raw
    r=where(var ne missing,c)
    dmax=3.0;max(var[r])
    dmin=0.0;min(var[r])
    p2=plot(julian_day,var,/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='fad_raw',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    t1=text(pos[pnum,0]+2.0*dx,pos[pnum,3]-2.0*dy,'FAD=liq_mwr/adiab_lwp',font_size=fs1)  

    ;***
    ; Fad param
    ;***

    pnum=4
    fad_param=make_array(/float,numtimes,value=missing) 
    r=where(cloud_layer_thickness le 250,c)
    if c gt 0 then fad_param[r]=0.8
    r=where(cloud_layer_thickness le 100,c)
    if c gt 0 then fad_param[r]=1.0
    r=where(cloud_layer_thickness gt 250,c)
    if c gt 0 then fad_param[r]=0.52-(0.00046*cloud_layer_thickness[r]) 
    var=fad_param
    r=where(fad_raw eq missing,c)
    if c gt 0 then var[r]=missing
    r=where(var ne missing,c)
    dmax=3.0;max(var[r])
    dmin=0.0;min(var[r])
    p2=plot(julian_day,var,/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='fad_param',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    t1=text(pos[pnum,0]+2.0*dx,pos[pnum,3]-2.0*dy,'FAD cloud layer thickness param',font_size=fs1)
    ax=p2.axes
    ax[3].Hide=1  ;hide right y axis
    var=cloud_layer_thickness
    r=where(fad_raw eq missing,c)
    if c gt 0 then var[r]=missing
    r=where(var ne missing,c)
    ;if c gt 0 then var[r]=var[r]*1e4
    dmin=min(var[r])
    dmax=max(var[r])
    p3=plot(julian_day,var,/current,position=pos[pnum,*],$
      symbol='o',linestyle=6,sym_size=0.1,yrange=[dmin,dmax],$
      xrange=[julian_day[0],julian_day[-1]],xtickunits='time',axis_style=4,color='red')
    p4=plot([julian_day[0],julian_day[-1]],[100,100],/overplot,color='red',linestyle=2)
    p4=plot([julian_day[0],julian_day[-1]],[250,250],/overplot,color='red',linestyle=2)

    yaxis=axis('Y',location='right',target=p3,title='layer thickness (m)',color='red')

    ;***************
    ;  Final FAD
    ;***************

    pnum=5
    var=fad
    r=where(var ne missing,c)
    dmax=3.0;max(var[r])
    dmin=0.0;min(var[r])
    p2=plot(julian_day,var,/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='fad',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    t1=text(pos[pnum,0]+2.0*dx,pos[pnum,3]-2.0*dy,'FAD final value',font_size=fs1)

    ;***************
    ;  Fad Source
    ;***************

    pnum=6
    var=fad_source
    r=where(fad eq missing,c)
    if c gt 0 then var[r]=missing
    r=where(var ne missing,c)
    dmax=4;max(var[r])
    dmin=0;min(var[r])
    p2=plot(julian_day,var,/current,position=pos[pnum,*],$
      xtickunits='hours',xtickinterval=2,xtickformat='(A1)',symbol='o',linestyle=6,sym_size=0.1,$
      ytitle='fad',xrange=[julian_day[0],julian_day[-1]],yrange=[dmin,dmax])
    t1=text(pos[pnum,0]+2.0*dx,pos[pnum,3]-2.0*dy,'FAD Source 1=liq/adiablwp 2=cloud thick param',font_size=fs1)

    t1=text(0.1,0.98,program+' '+date_str[ddd],font_size=14)
    imagename=date_str[ddd]+'_lidar_profile_quantities.'+avg_str+'.p3.png'
    p0.save,outdir+imagename
  endif  ;turn off plot 3


  ;****************************
  ;  Plot some individual profiles
  ;****************************

  ;  Turn on plotting
  if plot_flag eq 1 and num_to_avg eq 0.0 then begin

  ;  Choose which profiles to plot
  ;  Night profiles
  ;idx=where(solar_zenith ge 90,count_profiles)  
  ;  Low cloud during the night
  ;idx=where(solar_zenith ge 90 and cloud_base_ceil gt 0 and cloud_base_ceil le 4,count_profiles)
  ;  Have a subcloud dbz
  ;idx=where(dbz_avg_subcloud ne missing,count_profiles)
  ;  candidates for extinction
  ;idx=where(low_cloud_flag eq 1 and eta gt 0.0 and min_beta_abv_cldbase lt noise_abv_thresh,count_profiles)

  idx=where(low_cloud_flag eq 1 and nd_dlnbeta gt 0 and eta gt 0 and mean_zmax_array gt 0 and $
    min_beta_abv_cldbase lt noise_abv_thresh,count_profiles)
  ;idx=where(julian_day ge julday(3,1,2018,17,00,0) and cloud_base_ceil gt 0.5,count_profiles)  
 
  ;  Select one profile to plot
  ;idx=2489 & count_profiles=1
  for i=0,count_profiles-1 do begin
  ;for i=0,count_profiles-1,20 do begin
  ;if idx[i] ge 539 then begin

  caldat,julian_day[idx[i]],mm,dd,yy,hh,mi,ss
  print,yy,mm,dd,hh,mi,ss
  date=string(yy,format='(I4)')+string(mm,format='(I02)')+string(dd,format='(I02)')+$
    ' '+string(hh,format='(I02)')+':'+string(mi,format='(I02)')+':'+string(ss,format='(I02)')
  datehms_str=string(yy,format='(I4)')+string(mm,format='(I02)')+string(dd,format='(I02)')+$
    '.'+string(hh,format='(I02)')+string(mi,format='(I02)')+string(ss,format='(I02)')
  ymd_str=string(yy,format='(I4)')+string(mm,format='(I02)')+string(dd,format='(I02)')
  print, datehms_str
  
  ;***  Plot input ***
  pxdim=1000 & pydim=900
  xl=0.08 & xr=0.96
  yb=0.05 & yt=0.85
  sx=0.06
  sy=0.07
  numplots_x=3
  numplots_y=2
  dx=0.01 & dy=0.01
  position_plots,xl,xr,yb,yt,sx,sy,numplots_x,numplots_y,pos
  
  ;  copol plotting range
  copolmin=1e-4
  copolmax=1e3
  
  ;  Height index of estimated cloud base
  result=min(abs(cloud_base_estimate[idx[i]]-height),beidx)
  
  ;  Height index of cloud base
  closest=min(abs(cloud_base_ceil[idx[i]]-height),hidx)
  
  ;  Height index of cloud top
  if cloud_top_radar[idx[i]] eq -8888 then begin
    tidx=hidx+40
  endif else begin
    closest=min(abs(cloud_top_radar[idx[i]]-height),tidx)
    if tidx gt 400 then tidx=hidx+40
  endelse
  ;  Height indexes to plot
  shidx=hidx-40
  if shidx lt 0 then shidx=0
  ehidx=tidx+40
  if ehidx ge numheights then ehidx=numheights-1
    
  pnum=0  ;CLOUD MASK
  p0=plot(cloud_mask_wacr[idx[i],shidx:ehidx],height[shidx:ehidx],/buffer,position=pos[pnum,*],/ystyle,$
    xtickinterval=1,xminor=0,yrange=[height[shidx],height[ehidx]],$
    xrange=[-1,8],xtitle='wacr cloud mask',ytitle='height (km)',dimensions=[pxdim,pydim],symbol='o',sym_size=0.2)
  p3=plot([-1,8],[cloud_base_ceil[idx[i]],cloud_base_ceil[idx[i]]],/overplot,linestyle=0,color='blue')
  p3=plot([-1,8],[cloud_base_estimate[idx[i]],cloud_base_estimate[idx[i]]],/overplot,linestyle=2,color='medium sea green')
  p3=plot([-1,8],[cloud_top_sonde[idx[i]],cloud_top_sonde[idx[i]]],/overplot,linestyle=0,color='olive')
  p5=plot([-1,8],[cloud_top_radar[idx[i]],cloud_top_radar[idx[i]]],/overplot,linestyle=2,color='purple')
  p5=plot([-1,8],[cloud_base_thresh_height[idx[i]],cloud_base_thresh_height[idx[i]]],/overplot,linestyle=3,color='saddle brown')
  p3=plot([-1,8],[copol_max_height[idx[i]],copol_max_height[idx[i]]],/overplot,linestyle=3,color='orange')
  t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-4*dy,'0 missing')
  t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-6*dy,'1 clear or clutter')
  t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-8*dy,'2 rl cloud')
  t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-10*dy,'3 r only cloud')
  t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-12*dy,'4 l only cloud')  
  t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-14*dy,'5 r cloud, l miss')   
  t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-16*dy,'6 l cloud, r miss')  
  t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-18*dy,'7 clutter')  
  
  
  pnum=1  ;BACKSCATTER MPL LOG for Nd
  p3=plot(lidar_bscat_copol[idx[i],shidx:ehidx],height[shidx:ehidx],/current,position=pos[pnum,*],$
    /ystyle,xtitle='lidar_bscat (star=adjusted)',ytitle='height (km)',xrange=[copolmin,copolmax],$
    symbol='S',sym_size=0.4,yrange=[height[shidx],height[ehidx]],/xlog)
  p3=plot(lidar_bscat_crospol[idx[i],shidx:ehidx],height[shidx:ehidx],/overplot,linestyle=0,color='red',symbol='S',sym_size=0.4)
  ;p3=plot(lidar_bscat_copol_orig[idx[i],shidx:ehidx],height[shidx:ehidx],/overplot,linestyle=2,color='black',symbol='o',sym_size=0.3)
  ;p3=plot(lidar_bscat_crospol_orig[idx[i],shidx:ehidx],height[shidx:ehidx],/overplot,linestyle=2,color='red',symbol='o',sym_size=0.3)
  p3=plot([copolmin,copolmax],[cloud_base_ceil[idx[i]],cloud_base_ceil[idx[i]]],/overplot,linestyle=0,color='blue')  
  p3=plot([copolmin,copolmax],[cloud_base_estimate[idx[i]],cloud_base_estimate[idx[i]]],/overplot,linestyle=2,color='medium sea green')  
  p3=plot([copolmin,copolmax],[cloud_top_sonde[idx[i]],cloud_top_sonde[idx[i]]],/overplot,linestyle=0,color='olive')
  p5=plot([copolmin,copolmax],[cloud_top_radar[idx[i]],cloud_top_radar[idx[i]]],/overplot,linestyle=2,color='purple')   
  p5=plot([copolmin,copolmax],[cloud_base_thresh_height[idx[i]],cloud_base_thresh_height[idx[i]]],/overplot,linestyle=3,color='saddle brown')   
  p3=plot([copolmin,copolmax],[copol_max_height[idx[i]],copol_max_height[idx[i]]],/overplot,linestyle=3,color='orange')
  
  p5=plot([1,1],[copol_max_height[idx[i]]-(mean_zmax_array[idx[i]]/1000.0),copol_max_height[idx[i]]],/overplot,linestyle=0,color='gray',thick=3)     
  p4=plot([1e-3,1e-3],[cloud_base[idx[i]],cloud_top[idx[i]]],/overplot,linestyle=1,symbol='D',sym_thick=2)
  
  t1=text(pos[pnum,0]+2*dx,pos[pnum,3]-4*dy,'copol')
  t1=text(pos[pnum,0]+2*dx,pos[pnum,3]-6*dy,'crospol',color='red')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-4*dy,'Zmax(m)='+strcompress(mean_zmax_array[idx[i]],/remove_all),color='gray')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-6*dy,'fad='+strcompress(mean_fad_array[idx[i]],/remove_all),color='black')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-8*dy,'eta='+strcompress(mean_eta_array[idx[i]],/remove_all),color='black')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-10*dy,'gamma_l='+strcompress(gamma_l[idx[i]],/remove_all),color='black')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-12*dy,'Nd='+strcompress(nd_dlnbeta[idx[i]],/remove_all),color='black')
  closest=min(abs(ccn_julian_day-julian_day[idx[i]]),midx)
  if closest lt 0.5D then begin
    t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-14*dy,'ccn 0.5='+strcompress(ccn_0p5_med[midx],/remove_all),color='black')
    t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-16*dy,'ccn 0.2='+strcompress(ccn_0p2_med[midx],/remove_all),color='black')
  endif
  
  
  pnum=2  ;BACKSCATTER MPL LOG for extinction
  p3=plot(lidar_bscat_copol[idx[i],shidx:ehidx],height[shidx:ehidx],/current,position=pos[pnum,*],$
    /ystyle,xtitle='lidar_bscat (star=adjusted)',ytitle='height (km)',symbol='S',sym_size=0.4,$
    yrange=[height[shidx],height[ehidx]],/xlog,xrange=[copolmin,copolmax])
  p3=plot(lidar_bscat_crospol[idx[i],shidx:ehidx],height[shidx:ehidx],/overplot,linestyle=0,color='red',symbol='S',sym_size=0.4)  
  
  ;  Fit from dlnbeta_dz  
  lidar_bscat_copol1=lidar_bscat_copol[idx[i],shidx:ehidx]
  height1=height[shidx:ehidx]
  r=where(lidar_bscat_copol1 gt cloud_base_threshold,c)
  cbt_idx=r[0]-2
  p3=plot( [exp((dlnbeta_dz_fit[idx[i],1]*(height[shidx+cbt_idx]))+dlnbeta_dz_fit[idx[i],0]),$
            exp((dlnbeta_dz_fit[idx[i],1]*(height[hidx+1]))+dlnbeta_dz_fit[idx[i],0])],$
           [height[shidx+cbt_idx],height[hidx+1]],color='medium sea green',/overplot,thick=1)    
                
  closest=min(abs(height-dlnbeta_dz_fit_height[idx[i],0]),idx0)
  closest=min(abs(height-dlnbeta_dz_fit_height[idx[i],1]),idx1)
  p3=plot([lidar_bscat_copol[idx[i],idx0],lidar_bscat_copol[idx[i],idx1]],$
          [dlnbeta_dz_fit_height[idx[i],0],dlnbeta_dz_fit_height[idx[i],1]],$
          color='medium sea green',/overplot,linestyle=6,symbol='S',sym_size=0.5)
            
  ;  Fit from extinction_via_slope          
  p4=plot([exp((ext_via_slope_fit[idx[i],1]*(height[hidx-2]))+ext_via_slope_fit[idx[i],0]),$
           exp((ext_via_slope_fit[idx[i],1]*(height[hidx+8]))+ext_via_slope_fit[idx[i],0])],$
           [height[hidx-2],height[hidx+8]],color='deep sky blue',/overplot,thick=1)  
  
  lidar_heights_fit1=lidar_heights_fit[idx[i],*]
  copol_fit1=copol_fit[idx[i],*]
  r=where(lidar_heights_fit1 ne missing,c)
  if c gt 0 then begin
    lidar_heights_fit1=lidar_heights_fit1[r]
    copol_fit1=copol_fit1[r]
    p4=plot(exp(copol_fit1),lidar_heights_fit1,color='deep sky blue',/overplot,linestyle=6,symbol='S',sym_size=0.5)
  endif
  p7=plot([noise_abv_thresh,noise_abv_thresh],[height[shidx],height[ehidx]],/overplot,linestyle=3)
  p7=plot([cloud_base_threshold,cloud_base_threshold],[height[shidx],height[ehidx]],$
    /overplot,linestyle=2,color='medium sea green')
  p3=plot([copolmin,copolmax],[cloud_base_ceil[idx[i]],cloud_base_ceil[idx[i]]],/overplot,linestyle=0,color='blue')  
  p3=plot([copolmin,copolmax],[cloud_base_estimate[idx[i]],cloud_base_estimate[idx[i]]],/overplot,linestyle=2,color='medium sea green')  
  p3=plot([copolmin,copolmax],[cloud_top_sonde[idx[i]],cloud_top_sonde[idx[i]]],/overplot,linestyle=0,color='olive')
  p5=plot([copolmin,copolmax],[cloud_top_radar[idx[i]],cloud_top_radar[idx[i]]],/overplot,linestyle=2,color='purple')   
  p5=plot([copolmin,copolmax],[cloud_base_thresh_height[idx[i]],cloud_base_thresh_height[idx[i]]],/overplot,linestyle=3,color='saddle brown')   
  p3=plot([copolmin,copolmax],[copol_max_height[idx[i]],copol_max_height[idx[i]]],/overplot,linestyle=3,color='orange')
  t1=text(pos[pnum,0]+2*dx,pos[pnum,3]-4*dy,'copol')
  t1=text(pos[pnum,0]+2*dx,pos[pnum,3]-6*dy,'crospol',color='red')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-4*dy,'dlnbeta_dz_fit',color='medium sea green')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-6*dy,'extinction_fit',color='deep sky blue')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-8*dy,'ext slope='+strcompress(ext_via_slope_fit[idx[i],1],/remove_all),color='black')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-10*dy,'eta='+strcompress(eta[idx[i]],/remove_all),color='black')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-12*dy,'chsqr='+strcompress(chsqr_ext[idx[i]],/remove_all),color='black')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]-14*dy,'extinction='+strcompress(extinction_slope[idx[i]]),color='black')
 
  if 1 eq 0 then begin
    pnum=2  ;BACKSCATTER MPL LINEAR
    p3=plot(lidar_bscat_copol_orig[idx[i],shidx:ehidx],height[shidx:ehidx],/current,position=pos[pnum,*],$
      /ystyle,xtitle='lidar_bscat (dash=raw)',ytitle='height (km)',xrange=[-0.02,0.06],linestyle=2)
    p3=plot(lidar_bscat_crospol_orig[idx[i],shidx:ehidx],height[shidx:ehidx],/overplot,color='red',linestyle=2)
    p3=plot(lidar_bscat_copol[idx[i],shidx:ehidx],height[shidx:ehidx],/overplot,color='black')
    p3=plot(lidar_bscat_crospol[idx[i],shidx:ehidx],height[shidx:ehidx],/overplot,color='red')
    p3=plot([-2,2],[cloud_base_ceil[idx[i]],cloud_base_ceil[idx[i]]],/overplot,linestyle=0,color='blue') 
    p3=plot([0,0],[height[shidx],height[ehidx]],/overplot,linestyle=1,color='black')
    p3=plot([-2,2],[cldbase_atten_back_ht[idx[i]],cldbase_atten_back_ht[idx[i]]],/overplot,linestyle=0,color='green') 
    p3=plot([-2,2],[cldbase_atten_back_base[idx[i]],cldbase_atten_back_base[idx[i]]],/overplot,linestyle=2,color='green') 
    p3=plot([-2,2],[cldbase_atten_back_top[idx[i]],cldbase_atten_back_top[idx[i]]],/overplot,linestyle=2,color='green') 
    t1=text(pos[pnum,0]+0*dx,pos[pnum,3]+1*dy,'cldbase_atten_back='+string(cldbase_atten_back[idx[i]]),color='green')
    t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-4*dy,'copol')
    t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-6*dy,'crospol',color='red')
    t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-8*dy,'ceil base',color='blue') 
    t1=text(pos[pnum,0]+15*dx,pos[pnum,3]-10*dy,'max copol',color='green') 
  endif
  
  pnum=3  ;SOUNDING
  p1=plot(temp[idx[i],shidx:ehidx],height[shidx:ehidx],/current,position=pos[pnum,*],$
    yrange=[height[shidx],height[ehidx]],$
    ystyle=1,xtitle='Sonde Temp (C)',ytitle='height (km)',xrange=[-40,0],symbol='o',sym_size=0.2)
  p2=plot(dpt[idx[i],shidx:ehidx],height[shidx:ehidx],/overplot,color='red')
  p3=plot([-40,0],[cloud_base_ceil[idx[i]],cloud_base_ceil[idx[i]]],/overplot,linestyle=0,color='blue')
  p3=plot([-40,0],[cloud_base_estimate[idx[i]],cloud_base_estimate[idx[i]]],/overplot,linestyle=2,color='medium sea green')
  p3=plot([-40,0],[cloud_top_sonde[idx[i]],cloud_top_sonde[idx[i]]],/overplot,linestyle=0,color='olive')
  p5=plot([-40,0],[cloud_top_radar[idx[i]],cloud_top_radar[idx[i]]],/overplot,linestyle=2,color='purple')
  p5=plot([-40,0],[cloud_base_thresh_height[idx[i]],cloud_base_thresh_height[idx[i]]],/overplot,linestyle=3,color='saddle brown')
  p3=plot([-40,0],[copol_max_height[idx[i]],copol_max_height[idx[i]]],/overplot,linestyle=3,color='orange')
  ax=p1.axes
  ax[2].Hide=1  ;hide top x
  p7=plot(rh[idx[i],shidx:ehidx],height[shidx:ehidx],/current,position=pos[pnum,*],axis_style=4,$
    xrange=[0,100],color='teal',yrange=[height[shidx],height[ehidx]])
  p8=plot([70,70],[height[shidx],height[ehidx]],/overplot,color='teal',linestyle=2,clip=0)
  xaxis=axis('X',location='top',target=p7,title='RH (%)',color='teal');,tickfont_size=fs1)
  t1=text(pos[pnum,0]+2*dx,pos[pnum,3]-3*dy,'dewpoint',color='red')
  t1=text(pos[pnum,0]+2*dx,pos[pnum,3]-5*dy,'sonde dt='+string(sonde_dt[idx[i]],format='(F8.2)')+'min',color='black')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]+3*dy,'radar top='+string(cloud_top_temp[idx[i]],format='(F8.2)'),color='purple')
  t1=text(pos[pnum,0]+0*dx,pos[pnum,3]+3*dy,'ceil base='+string(cloud_base_temp[idx[i]],format='(F8.2)'),color='blue')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,3]+5*dy,'sonde top='+string(cloud_top_sonde_temp[idx[i]],format='(F8.2)'),color='olive')
  t1=text(pos[pnum,0]-2*dx,pos[pnum,3]+5*dy,'estimate base='+string(temp[idx[i],beidx],format='(F8.2)'),color='medium sea green')
  
  pnum=4  ;DEPOLARIZATION
  p3=plot(lidar_bscat_crospol[idx[i],shidx:ehidx]/lidar_bscat_copol[idx[i],shidx:ehidx],height[shidx:ehidx],$
    /current,position=pos[pnum,*],yrange=[height[shidx],height[ehidx]],$
    ystyle=1,xtitle='depolarization ratio',ytitle='height (km)',xrange=[-1,1.5],symbol='o',sym_size=0.2) 
  if hidx ge 21 then begin
    sub_cloud_ht1=height[hidx-7-14:hidx-7]
    sub_cloud_depol1=reform(lidar_bscat_crospol[idx[i],hidx-7-14:hidx-7]/lidar_bscat_copol[idx[i],hidx-7-14:hidx-7])
    sub_cloud_mask1=reform(cloud_mask_wacr[idx[i],hidx-7-14:hidx-7])
    r=where(sub_cloud_mask1 eq 2,c)
    if c gt 0 then $
      p6=plot(sub_cloud_depol1[r],sub_cloud_ht1[r],/overplot,color='deep pink',symbol='o',sym_size=0.2,/sym_filled,linestyle=6)
    if depol_below[idx[i]] ne -8888 then $
      p4=plot([depol_below[idx[i]],depol_below[idx[i]]],$
        [height[hidx-7-14],height[hidx-7]],/overplot,color='deep pink',thick=1)
  endif
  p3=plot([-1,2],[cloud_base_ceil[idx[i]],cloud_base_ceil[idx[i]]],/overplot,linestyle=0,color='blue')
  p3=plot([-1,2],[cloud_base_estimate[idx[i]],cloud_base_estimate[idx[i]]],/overplot,linestyle=2,color='medium sea green')
  p3=plot([-1,2],[cloud_top_sonde[idx[i]],cloud_top_sonde[idx[i]]],/overplot,linestyle=0,color='olive')
  p5=plot([-1,2],[cloud_top_radar[idx[i]],cloud_top_radar[idx[i]]],/overplot,linestyle=2,color='purple')
  p5=plot([-1,2],[cloud_base_thresh_height[idx[i]],cloud_base_thresh_height[idx[i]]],/overplot,linestyle=3,color='saddle brown')
  p3=plot([-1,2],[copol_max_height[idx[i]],copol_max_height[idx[i]]],/overplot,linestyle=3,color='orange')   
  t1=text(pos[pnum,0]+0*dx,pos[pnum,3]+1*dy,'ceil base depol='+string(cloud_base_depol[idx[i]],format='(F10.4)'),color='blue') 
  t1=text(pos[pnum,0]+0*dx,pos[pnum,3]+3*dy,'depol_below='+string(depol_below[idx[i]],format='(F10.4)'),color='deep pink')
  t1=text(pos[pnum,0]+13*dx,pos[pnum,1]+2*dy,'adiab_lwp(g/m2)='+string(adiab_lwp[idx[i]]*1e3,format='(F6.2)'))       
  t1=text(pos[pnum,0]+13*dx,pos[pnum,1]+4*dy,'layer(m)='+string(cloud_layer_thickness[idx[i]],format='(F6.2)'))    
  t1=text(pos[pnum,0]+13*dx,pos[pnum,1]+6*dy,'fad_source='+string(fad_source[idx[i]],format='(I1)'))
  t1=text(pos[pnum,0]+13*dx,pos[pnum,1]+8*dy,'fad_raw='+string(fad_raw[idx[i]],format='(F10.4)'))
  
  pnum=5  ;RADAR
  p3=plot(dbz_wacr[idx[i],shidx:ehidx],height[shidx:ehidx],/current,position=pos[pnum,*],$
    xrange=[-70,20],yrange=[height[shidx],height[ehidx]],$
    ystyle=1,xtitle='wacr dbz',ytitle='height (km)',symbol='o',sym_size=0.2,color='red') 
  p3=plot(dbz_wacr_orig[idx[i],shidx:ehidx],height[shidx:ehidx],/overplot,color='black')   
  p3=plot([radar_top_threshold[idx[i]],radar_top_threshold[idx[i]]],[height[shidx],height[ehidx]],/overplot,linestyle=2,color='red')
  p3=plot([-70,20],[cloud_base_ceil[idx[i]],cloud_base_ceil[idx[i]]],/overplot,linestyle=0,color='blue')
  p3=plot([-70,20],[cloud_base_estimate[idx[i]],cloud_base_estimate[idx[i]]],/overplot,linestyle=2,color='medium sea green')
  p3=plot([-70,20],[cloud_top_sonde[idx[i]],cloud_top_sonde[idx[i]]],/overplot,linestyle=0,color='olive')
  p5=plot([-70,20],[cloud_top_radar[idx[i]],cloud_top_radar[idx[i]]],/overplot,linestyle=2,color='purple')
  p5=plot([-70,20],[cloud_base_thresh_height[idx[i]],cloud_base_thresh_height[idx[i]]],/overplot,linestyle=3,color='saddle brown')
  p3=plot([-70,20],[copol_max_height[idx[i]],copol_max_height[idx[i]]],/overplot,linestyle=3,color='orange')
  if hidx ge 21 then begin
    sub_cloud_ht1=height[hidx-7-14:hidx-7]
    sub_cloud_dbz1=reform(dbz_wacr[idx[i],hidx-7-14:hidx-7])
    sub_cloud_mask1=reform(cloud_mask_wacr[idx[i],hidx-7-14:hidx-7])
    r=where(sub_cloud_mask1 eq 2,c)
    if c gt 0 then $
    p6=plot(sub_cloud_dbz1[r],sub_cloud_ht1[r],/overplot,color='teal',symbol='o',sym_size=0.2,/sym_filled,linestyle=6)  
    if dbz_avg_subcloud[idx[i]] ne -8888 then $  
    p4=plot([dbz_avg_subcloud[idx[i]],dbz_avg_subcloud[idx[i]]],$
      [height[hidx-7-14],height[hidx-7]],/overplot,color='teal',thick=1)
  endif
  t1=text(pos[pnum,0]+0*dx,pos[pnum,3]+1*dy,'ceil base dbz='+string(cldbase_dbz[idx[i]],format='(F10.2)'),color='blue')
  t1=text(pos[pnum,0]+0*dx,pos[pnum,3]+3*dy,'dbz_avg_subcloud='+string(dbz_avg_subcloud[idx[i]],format='(F10.2)'),color='teal')
  t1=text(pos[pnum,0]+0*dx,pos[pnum,3]+5*dy,'dbz_max_depol='+string(dbz_max_depol[idx[i]],format='(F10.2)'),color='gray')
  t1=text(pos[pnum,0]+0*dx,pos[pnum,3]+7*dy,'liq(g/m2)='+string(liq_mwr[idx[i]]*1e4,format='(F17.5)'),color='dodger blue')
  t1=text(pos[pnum,0]+0*dx,pos[pnum,3]+9*dy,'be_lwp(g/m2)='+string(be_lwp[idx[i]],format='(F10.5)'),color='steel blue')
  t1=text(pos[pnum,2]-12*dx,pos[pnum,3]-3*dy,'radar')
  t1=text(pos[pnum,2]-12*dx,pos[pnum,3]-5*dy,'radar+5.0dB',color='red')
  
  
  t1=text(0.1,0.97,datehms_str,font_size=14)
  t1=text(0.3,0.97,'solar zenith angle='+string(solar_zenith[idx[i]],format='(F7.2)'),font_size=14 )
  t1=text(0.7,0.97,'lat='+string(lat[idx[i]],format='(F6.2)'),font_size=14)
  t1=text(0.84,0.97,'lon='+string(lon[idx[i]],format='(F6.2)'),font_size=14)
  
  t1=text(pos[0,0]+0*dx,pos[0,3]+1*dy,'estimate base',color='medium sea green',font_size=fs2) 
  t1=text(pos[0,0]+12*dx,pos[0,3]+1*dy,'ceil base',color='blue',font_size=fs2)
  t1=text(pos[0,0]+21*dx,pos[0,3]+1*dy,'radar top',color='purple',font_size=fs2)
  t1=text(pos[0,0]+29*dx,pos[0,3]+1*dy,'sonde top',color='olive',font_size=fs2) 
  t1=text(pos[0,0]+38*dx,pos[0,3]+1*dy,'thresh base',color='saddle brown',font_size=fs2) 
  t1=text(pos[0,0]+48*dx,pos[0,3]+1*dy,'max copol height',color='orange',font_size=fs2) 
  t1=text(pos[2,0]+2*dx,pos[2,3]+1*dy,'attenuation threshold='+string(noise_abv_thresh,format='(F8.6)'))
  t1=text(pos[2,0]+2*dx,pos[2,3]+2.5*dy,'cloud base threshold='+string(cloud_base_threshold,format='(F4.1)'))
  
  image_name='lidar_profile.'+datehms_str+'.'+string(idx[i],format='(I05)')+'.avg30s.v2023.png'
  p0.save,outdir+image_name;,height=pydim  
  print,image_name
  if i mod 50 eq 0 then stop
 
 
    endfor  ;end of loop through profiles to plot
  endif  ;turn on profile plot
endif   ; if max(low_cloud_flag) eq 1 then begin
endif ; if max(wind_filter) gt 0. then begin
endif ; if count_files_30s gt 0 and count_smet gt 0 and count_ccn gt 0 then begin
endfor  ;end of loop through list of dates
;stop
end